function [img,mmPerPix,volSize,fileName] = readVolAnat(fileName)
% [img,mmPerPix,volSize,fileName] = readVolAnat([fileName])
%
% Loads the vAnatomy.dat file specified by fileName (full path!)
% into the [rows,cols,planes] image cube 'img'.
%
% If fileName is omitted, a get file dialog appears.
%
% RETURNS:
%   * img is the [rows,cols,planes] intensity array
%   * mmPerPix is the voxel size (in mm/pixel units)
%   * fileName is the full-path to the vAnatomy.dat file. (If 
%     you pass fileName in, you obviously don't need this. But 
%     it may be useful when the user selects the file.)
%
% 2000.01.28 RFD
% 2001.02.21 RFD: modified it to try the UnfoldParams.mat if the 
%            mmPerPix was not found in the vAnatomy header. It also
%            now returns the full path with filename, rather than
%            just the directory.

% set this to nan's incase the vAnatomy is old-style, in which
% case it won't have mmPerPix in the header.
mmPerPix = [nan,nan,nan];

if ~exist('fileName', 'var')
   fileName = '';
end

% if fileName is empty, get the filename and path with a uigetfile
if isempty(fileName)
	[fname, fpath] = uigetfile('*.dat', 'Select a vAnatomy file...');
   fileName = [fpath fname];
   if fname == 0
      % user cancelled
      img = [];
      return;
   end
else
   [fpath,fname,ext] = fileparts(fileName);
   fname = [fname ext];
end

% open file for reading (little-endian mode)
vFile = fopen(fileName,'r');
if vFile==-1
   img = [];
   myErrorDlg(['Couldn''t open ',fileName,'!'])
   return;
end

volSize = [nan,nan,nan];

tmp = fscanf(vFile,'rows %f (%f mm/pixel)\n');
volSize(1) = tmp(1);
if length(tmp)>1 mmPerPix(1) = tmp(2); end;
tmp = fscanf(vFile,'cols %f (%f mm/pixel)\n');
volSize(2) = tmp(1);
if length(tmp)>1 mmPerPix(2) = tmp(2); end;
tmp = fscanf(vFile,'planes %f (%f mm/pixel)\n');
volSize(3) = tmp(1);
if length(tmp)>1 mmPerPix(3) = tmp(2); end;

endOfHeader = '*';
nextLine = fgets(vFile);
if ~(length(nextLine==2) & nextLine(1)==endOfHeader)
  myErrorDlg(['vAnatomy file: ',fileName,' has invalid header']);
end
[vData cnt] = fread(vFile,prod(volSize),'uchar');
fclose(vFile);

if(isnan(mmPerPix(1)) | isnan(mmPerPix(2)) | isnan(mmPerPix(3)))
    % Try to get the voxel size from the old UnfoldParams.mat file.
	disp('Looking for voxel size in UnfoldParams.mat file');
	
    eval('ufp = load(fullfile(fpath,''UnfoldParams.mat''));', 'ufp = [];');
    if(isfield(ufp, 'volume_pix_size'))
        mmPerPix = 1./ufp.volume_pix_size;
	else
        % mmPerPix can not be discerned- warn user?
		disp ('Warning! No mmPerPix! Assuming  (1.06666666666667   1.066666667 1.21)');
		mmPerPix=[1.0666666666 1.0666666666666 1.2097];
	end
	

end

% *** HACK!  Sometimes the vAnatomy is missing the last byte (???)
if length(vData) == prod(volSize)-1
   vData(end+1) = 0;
end

img = zeros(volSize);
% transpose each slice to switch rows and cols
slicePix = volSize(1)*volSize(2);
for ii=0:volSize(3)-1
   img(:,:,ii+1) = reshape(vData(ii*slicePix+1:ii*slicePix+slicePix),[volSize(2) volSize(1)])';
end

return
