function spm_defaults_edit(varargin)
% Modify defaults
% FORMAT spm_defaults_edit
%_______________________________________________________________________
%
% spm_defaults_edit allows the current defaults to be edited.
%
% These changes do not persist across sessions. SPMs startup defaults
% are specified in the first spm_defaults on the MATLABPATH.
%
% The defaults which can be modified are:
% 
% Printing Options
%     Allows a number of different printing defaults to be specified.
% 
% Miscellaneous Defaults
%     This includes:
%     * Specification of a file for logging dialogue between
%       the user and SPM.
%     * Command line input option. Rather than clicking
%       buttons on the interface, input can be typed to
%       the Matlab window.
%     * The intensity of any grid which superimposed on any
%       displayed images.
% 
% Realignment & Coregistration Defaults
%     An assortment of defaults.
%
% Spatial Normalisation Defaults
%     An assortment of defaults.
%
% The 'reset' option re-loads the startup defaults from spm_defaults.m
%
%_______________________________________________________________________
% @(#)spm_defaults_edit.m	2.25 John Ashburner 03/01/27
SCCSid = '2.25';

%-Format arguments
%-----------------------------------------------------------------------
if nargin == 0, Action='!EditMenu'; else, Action = varargin{1}; end


%-Get/setup global defaults variables
%-----------------------------------------------------------------------
global defaults

switch lower(Action), case '!editmenu'                    %-Defaults menu
%=======================================================================
	SPMid = spm('FnBanner',mfilename,SCCSid);
	spm('FnUIsetup','Defaults Edit');
	spm_help('!ContextHelp',mfilename)

	callbacks = strvcat(...
		'spm_defaults_edit(''Printing'');',...
		'spm_defaults_edit(''Misc'');',...
		'spm_realign_ui(''Defaults'');',...
                'spm_realign_ui(''UnwarpDefaults'');',...
		'spm_coreg_ui(''Defaults'');',...
		'spm_normalise_ui(''Defaults'');',...
		'spm_segment_ui(''Defaults'');',...
		'spm_bias_ui(''Defaults'');',...
		'spm_defaults_edit(''Statistics'');',...
		'spm_defaults_edit(''Reset'');'...
		);

	a1 = spm_input('Defaults Area?',1,'m',...
		['Printing Options|'...
		 'Miscellaneous Defaults|'...
		 'Realignment|'...
                 'Unwarp|'...
		 'Coregistration|'...
		 'Spatial Normalisation|'...
		 'Segmentation|'...
		 'Bias Correction|'...
		 'Statistics - ',defaults.modality,'|'...
		 'Reset All']);

	eval(deblank(callbacks(a1,:)));
	spm_figure('Clear','Interactive');

case 'realign'                          %-Realignment defaults
%=======================================================================

	spm_realign_ui('Defaults');

case 'coreg'                            %-Coreg defaults
%=======================================================================

	spm_coreg_ui('Defaults');

case 'normalisation'                    %-Spatial normalisation defaults
%=======================================================================

	spm_normalise_ui('Defaults');

case 'segmentation'                     %-Segmentation defaults
%=======================================================================
	spm_segment_ui('Defaults');

case 'bias'
%=======================================================================
	spm_bias_ui('Defaults');

case 'misc'                             %-Miscellaneous defaults
%=======================================================================

	%-Store CMDLINE setting
	c = (abs(defaults.cmdline)>0) -1;

	if ~isempty(defaults.logfile), tmp='yes'; def=1; else, tmp='no'; def=2; end
	if spm_input(['Log to file? (' tmp ')'],2*c,'y/n',[1,0],def)
		defaults.logfile = ...
			deblank(spm_input('Logfile Name:',2,'s', defaults.logfile));
	else
		defaults.logfile = '';
	end

	defaults.cmdline = abs(defaults.cmdline)>0 * sign(defaults.cmdline);
	def = find(defaults.cmdline==[0,1,-1]);
	defaults.cmdline = spm_input('Command Line Input?',3*c,'m',...
		{	'always use GUI',...
			'always use CmdLine',...
			'GUI for files, CmdLine for input'},...
		[0,1,-1],def);

	defaults.analyze.multivol = spm_input('Allow multi-volume Analyze files?',4*c,'y/n',...
		[1,0],2-defaults.analyze.multivol);

	defaults.grid = spm_input('Grid value (0-1):', 5*c, 'e', defaults.grid);


case 'printing'                                      %-Printing defaults
%=======================================================================

	a0 = spm_input('Printing Mode?', 2, 'm', [...
			'Postscript to File|'...
			'Postscript to Printer|'...
			'Other Format to File|'...
			'Custom']);

	if (a0 == 1)
		fname = date; fname(find(fname=='-')) = []; fname = ['spmfig_' fname];
		fname = spm_str_manip(spm_input('Postscript filename:',3,'s',fname),'rtd');

		a1    = spm_input('Postscript Type?', 4, 'm', [...
			'PostScript for black and white printers|'...
			'PostScript for colour printers|'...
			'Level 2 PostScript for black and white printers|'...
			'Level 2 PostScript for colour printers|'...
			'Encapsulated PostScript (EPSF)|'...
			'Encapsulated Colour PostScript (EPSF)|'...
			'Encapsulated Level 2 PostScript (EPSF)|'...
			'Encapsulated Level 2 Color PostScript (EPSF)|'...
			'Encapsulated                with TIFF preview|'...
			'Encapsulated Colour         with TIFF preview|'...
			'Encapsulated Level 2        with TIFF preview|'...
			'Encapsulated Level 2 Colour with TIFF preview|']);

		prstr1 = strvcat(...
			['print(''-noui'',''-painters'',''-dps'' ,''-append'',''' fname '.ps'');'],...
			['print(''-noui'',''-painters'',''-dpsc'',''-append'',''' fname '.ps'');'],...
			['print(''-noui'',''-painters'',''-dps2'',''-append'',''' fname '.ps'');'],...
			['print(''-noui'',''-painters'',''-dpsc2'',''-append'',''' fname '.ps'');']);
		prstr1 = strvcat(prstr1,...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-deps'',[''' fname '_'' num2str(PAGENUM) ''.ps'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-depsc'',[''' fname '_'' num2str(PAGENUM) ''.ps'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-deps2'',[''' fname '_'' num2str(PAGENUM) ''.ps'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-depsc2'',[''' fname '_'' num2str(PAGENUM) ''.ps'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-deps'',''-tiff'',[''' fname '_'' num2str(PAGENUM) ''.ps'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-depsc'',''-tiff'',[''' fname '_'' num2str(PAGENUM) ''.ps'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-deps2'',''-tiff'',[''' fname '_'' num2str(PAGENUM) ''.ps'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-depsc2'',''-tiff'',[''' fname '_'' num2str(PAGENUM) ''.ps'']); PAGENUM = PAGENUM + 1;']);
		defaults.printstr = deblank(prstr1(a1,:));
	elseif (a0 == 2)
		printer = '';
		if (spm_input('Default Printer?', 3, 'y/n') == 'n')
			printer = spm_input('Printer Name:',3,'s')
			printer = [' -P' printer];
		end
		a1 = spm_input('Postscript Type:',4,'b','B & W|Colour', ...
                         strvcat('-dps', '-dpsc'));
		defaults.printstr = ['print -noui -painters ' a1 printer];
	elseif (a0 == 3)
		fname = date; fname(find(fname=='-')) = []; fname = ['spmfig_' fname];
		fname = spm_str_manip(spm_input('Graphics filename:',3,'s', fname),'rtd');
		a1    = spm_input('Graphics Type?', 4, 'm', [...
			'HPGL compatible with Hewlett-Packard 7475A plotter|'...
			'Adobe Illustrator 88 compatible illustration file|'...
			'M-file (and Mat-file, if necessary)|'...
			'Baseline JPEG image|'...
			'TIFF with packbits compression|'...
			'Color image format|']);
		prstr1 = strvcat(...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-dhpgl'',[''' fname '_'' num2str(PAGENUM) ''.hpgl'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-dill'',[''' fname '_'' num2str(PAGENUM) ''.ill'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-dmfile'',[''' fname '_'' num2str(PAGENUM) ''.m'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-djpeg'',[''' fname '_'' num2str(PAGENUM) ''.jpeg'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-dtiff'',[''' fname '_'' num2str(PAGENUM) ''.tiff'']); PAGENUM = PAGENUM + 1;'],...
			['global PAGENUM;if isempty(PAGENUM),PAGENUM = 1;end;'...
			 'print(''-noui'',''-painters'',''-dtiffnocompression'',[''' fname '_'' num2str(PAGENUM) ''.tiff'']); PAGENUM = PAGENUM + 1;']);
		defaults.printstr = deblank(prstr1(a1,:));
	else
		defaults.printstr = spm_input('Print String',3,'s');
	end

case 'statistics'                                       %-Stats defaults
%=======================================================================

	if strcmp(defaults.modality,'PET'),
		defaults.stats.pet.ufp = spm_input('Upper tail F prob. threshold (PET)',...
			2,'e',defaults.stats.pet.ufp,1);
	elseif strcmp(defaults.modality,'FMRI')
		defaults.stats.fmri.ufp  = spm_input('Upper tail F prob. threshold (FMRI)',...
			2,'e',defaults.stats.fmri.ufp,1);
		defaults.stats.fmri.t  = spm_input('Number of Bins/TR',...
			3,'n',defaults.stats.fmri.t,1);
		defaults.stats.fmri.t0    = spm_input('Sampled bin',...
			4,'n',defaults.stats.fmri.t0,1, defaults.stats.fmri.t);
	end;


case 'reset'                                            %-Reset defaults
%=======================================================================
	if exist('spm_defaults')==2
		spm_defaults;
	end
      	spm('chmod',defaults.modality);

otherwise
%=======================================================================
error(['Invalid type/action: ',Action])


%=======================================================================
end

