function M = meshMovie(movieFileName,timeSteps,saveFlag)
%
%   M = meshMovie(movieFileName,timeSteps,saveFlag)
%
%Author: Wandell
%Purpose:
%   Create a movie of the fundamental component of the time series based on
%   the coherence and phase measurement. 
%   This is not the real time series, but just a signal-approximation.
%   At some point, we should read in the time series and show it frame by
%   frame.  I am not quite sure how to do this.  We don't normally get a
%   time-slice over space.  But I am guessing we could do it.
% 
%   The movie is made within an ROI that is specified in the open3dWindow.
%   Set the cursor to a position and set the ROI size to a radius.  Only
%   the activity in that area will be shown.
%
% Example:  To write out an AVI file called scratch, and to return a Matlab
% movie structure, M, within the constraints of the cothresh and ROI
% parameters, use: 
%
%     M = meshMovie('scratch',10);
%
% TODO:  Option for naming the output file
%        More instructions on how to use this file.
%


if ieNotDefined('saveFlag'),  saveFlag = 1; else saveFlag = 0; end
if ieNotDefined('timeSteps'), timeSteps = 10; end
if ieNotDefined('moveFileName'),movieFileName = 'Scratch'; end

global VOLUME;
selectedVOLUME = viewSelected('volume'); 
% We work on a copy of the selected volume
v = VOLUME{selectedVOLUME};

[saveFlag,timeSteps,movieFileName] = readParameters(saveFlag,timeSteps,movieFileName);
if isempty(saveFlag), return; end

% Empty out the ROIs except for the one we will make
v = viewSet(v,'rois',[]);
v = viewSet(v,'selectedroi',0);

% Make sure the cor anal data are loaded
if isempty(viewGet(v,'co')), v=loadCorAnal(v); end

msh = viewGet(v,'currentmesh');
pos = round(mrmGet(msh,'cursor'));
if max(pos(:)) > 255, error('Problem reading cursor position.  Click and try again.'); end

% Build an ROI of the right size.
roiName = sprintf('mrm-%.0f-%.0f-%.0f',pos(1),pos(2),pos(3));
handles = guihandles(gcbo);
roiRadius = str2double(get(handles.editROISize,'String'));
v = makeROIdiskGray(v,roiRadius,roiName,[],[],pos);

% Create a view with the ROI defined.  It will sit in the window for a
% moment.
[v,roiMask,junk,roiAnatColors] = meshColorOverlay(v,0);
if sum(roiMask) == 0, error('Bad roiMask'); end
msh = mrmSet(msh,'colors',roiAnatColors');

% This is where we will store the new color information
newColors = zeros(size(roiAnatColors));

% Set up the co or amp values for the movie.  We replace the colors within
% the dataMask with the new colors generated here.
curScan = viewGet(v,'currentscan');
realCO = viewGet(v, 'scanco',curScan);
ph = viewGet(v, 'scanph',curScan);

t = ([0:(timeSteps-1)]/timeSteps)*2*pi;
nFrame = length(t);

clear M;
mrmSet(msh,'hidecursor');

str = sprintf('Creating %.0f frame movie',nFrame);
wbar = waitbar(0,str);

v = setDisplayMode(v,'co');
for ii=1:nFrame
    str = sprintf('Creating frame %.0f of %.0f',ii,nFrame);
    fname{ii} = sprintf('Movie%0.4d.tiff',ii);
    waitbar(ii/nFrame,wbar,str);
    
    data = realCO.*(1 + sin(t(ii) - ph))/2;
    v = viewSet(v,'scancoherence',data,curScan);

    [v,junk,foo,newColors] = meshColorOverlay(v,0);
    msh = mrmSet(msh,'colors',newColors');
    
    roiAnatColors(1:3,logical(roiMask)) = newColors(1:3,logical(roiMask));
    msh = mrmSet(msh,'colors',roiAnatColors');
    
    img = mrmGet(msh,'screenshot')/255;
    M(ii) = im2frame(img);

end

% figure; axis image; axis off; movie(M,5,4)
waitbar(1,wbar)
close(wbar);

h = figure; set(h,'userdata',M); movie(M,5,4)

if saveFlag
    fprintf('Saving movie as avi file: %s\n',[pwd,filesep,movieFileName]);
    if(isunix)
        movie2avi(M,movieFileName,'FPS',3,'compression', 'none');
    else
        movie2avi(M,movieFileName,'FPS',3,'QUALITY',100,'compression', 'Indeo5'); 
    end
end

return;

%------------------------------------
function [saveFlag,timeSteps,movieFileName] = readParameters(saveFlag,timeSteps,movieFileName);
%
% read parameters for meshMovie
%

prompt={'Save File (0/1)?','Movie File name','Time Steps'};
def={num2str(saveFlag),movieFileName,num2str(timeSteps)};
dlgTitle='Enter Movie parameters';
lineNo=1;
answer=inputdlg(prompt,dlgTitle,lineNo,def);
if isempty(answer), saveFlag = []; return; end

saveFlag = str2num(answer{1});
movieFileName = answer{2};
timeSteps = str2num(answer{3});

return;

