function images = meshCompareScans(V, scans, dts, angles, savePath, leg);
%
% Create mosaic images showing data from different scans / data
% types superimposed on the same mesh and view angles. 
%
% images = meshCompareMaps(<view, scans, dts, angles, savePath, leg>);
%
% This code requires that you have saved some view angles using the 
% function meshAngles. The code will set the input view (which should
% have an open mesh attached) to each of the specified input scans, 
% display the current data (which depends on the display mode: co, amp,
% ph, map) on the mesh, set the mesh to the specified angles, and
% take a snapshot. For each angle provided, the code will produce an
% output image which is a mosaic of the maps across all the scans and data
% types. 
%
% INPUT ARGUMENTS:
%   view: gray view w/ open mesh attached. <Default: selected gray.>
%
%   scans: vector of scan numbers from which to take the data.
%       <default: all scans in cur data type>
%
%   dts: cell array of data type names / vector of data type numbers. 
%       if only one provided, and multiple scans, will assume all the
%       scans come from the current data type. <default: cur dt>
%
%   angles: struct array of angles. See meshAngles. Can also be a numeric
%       vector specifying which of the saved angles to show, or a cell of
%       names of angles. <default: all saved mesh angles>
%   
%   savePath: if provided, will either save the *.png files to this
%       directory (if dir) or append images to a power point file (if
%       using Windows and savePath ends in *.ppt).
%
%   leg: optional legend: an image which will be attached at
%       the end of each mosaic image. <default: no legend>
%
%
%
% OUTPUT ARGUMENT:
%   images: cell of images, one for each input angle specified.
%
%
%
% ras, 02/02/06. I've been writing this code for 50 years, but it's still
% always the same day!
if notDefined('V'),         V = getSelectedGray;                end
if notDefined('scans'),     scans = 1:numScans(V);              end
if notDefined('dts'),       dts = V.curDataType;                end
if notDefined('savePath'),  savePath = 'Images';                end
if notDefined('leg'),       leg = [];                           end
if notDefined('angles')     
    angles = meshAngle('list', 'quiet', V);      
end



%%%%%make sure the dts list is a numeric array
if iscell(dts)
    for i = 1:length(dts),   tmp(i) = existDataType(dts{i});    end
    dts = tmp;
end

for i = length(dts)+1:length(scans), dts(i) = dts(i-1);         end

%%%%%allow angles to be names / indices of saved angles
if isnumeric(angles)
    allAngles = meshAngle('list', 'quiet', V);
    angles = allAngles(angles);
elseif iscell(angles)
    allAngles = meshAngle('list', 'quiet', V);
    angles = allAngles(cellfind({allAngles.name}, angles));
end
    

%%%%%initialize cell arrays for each image (corresponding to each 
%%%%%angle) for the main loop
for i = 1:length(angles), pics{i} = {}; end

%%%%%get the current mesh
msh = viewGet(V, 'currentMesh');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Main loop: go through each scan, put up the map, grab the image %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for i = 1:length(scans)
    if dts(i)~=V.curDataType, V = selectDataType(V, dts(i)); end
    V = viewSet(V, 'curScan', scans(i));
    
    meshColorOverlay(V);
    
    % take a picture of the mesh, with this map, at each angle
    for j = 1:length(angles)
        meshAngle('load', angles(j).name, V);
        pics{j}{i} = mrmGet(msh, 'screenshot') ./ 255;
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Process the screenshot pics into mosaic images %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for i = 1:length(pics)
    if notDefined('leg')
        images{i} = imageMontage(pics{i});
    else
        images{i} = imageMontage([pics{i} leg]);
    end
    
    % save if specified
    if ~isempty(savePath)
        [p f ext] = fileparts(savePath);
        if isequal(lower(ext), '.ppt') & ispc
            % paste into powerpoint presentation
            fig = figure; imshow(img);
            [ppt, op] = pptOpen(savePath);
            pptPaste(op,fig,'meta');
            pptClose(op,ppt,savePath);
            close(fig);
            fprintf('Pasted image in %s.\n', fname);            
        else
            if isequal(lower(ext), '.ppt'), 
                q = ['Sorry, can only export to PowerPoint files on ' ...
                     'Windows machines right now. Save images as ' ...
                     '*.png files instead?'];
                resp = questdlg(q, mfilename); 
                if ~isequal(resp, 'Yes'), return; end
            end
            mapName = V.mapName;
            if checkfields(V, 'ui', 'displayMode')
                switch V.ui.displayMode
                    case 'ph', mapName = 'Phase';
                    case 'amp', mapName = 'Amplitude';
                    case 'co', mapName = 'Coherence';
                    case 'map', mapName = V.mapName;
                end
            end
            fname = sprintf('%s_%s.png', mapName, angles(i).name);
            imgPath = fullfile(savePath, fname);
            imwrite(images{i}, imgPath, 'png');
            fprintf('Saved image %s.\n', imgPath);
        end
    end
end

return
