function [v, dataMask, dataMaskIndices, newColors] = meshColorOverlay(v,showData,roiPerimThickness)
%
%  [v, dataMask,dataMaskIndices,newColors] = ...
%          meshColorOverlay(v,showData,roiPerimThickness);
%
% Compute and show color overlays (phase, co, etc) of the mesh in the
% current 3D view (3dWindow). 
%
% v is a VOLUME{} structure
% showData (default = 1) indicates whether to put the new colors into
% the mrMesh window.  
%
%Example:
%  v = VOLUME{1};
%  meshColorOverlay(v);        % Just renders the data
%  
%  Returns value (and renders).  Can be used to build up images.
%  [junk, dm, dmIndices, newColors] = meshColorOverlay(VOLUME{1},0);
%
% Author: Ress (June, 2003)
%
% 09/2005 SOD: change to handle datasets with NaNs
% 10/2005 SOD: put in switches to handle phase data

if notDefined('v'), v = getSelectedGray; end
if notDefined('showData'), showData = 1; end
if notDefined('roiPerimThickness')
    % Option to display only the ROI perimeter. '1' is a good number to use here..
    % ras 01/06: inherits from the view's settings
    if checkfields(v, 'ui', 'showROIs') & (v.ui.showROIs < 0)
        roiPerimThickness = 1;
    else
        roiPerimThickness = 0;
    end
end
msh = viewGet(v,'currentmesh');

% NOTE: sometimes a view may be set to map, amp, co, or ph mode, but
% a parameter map or corAnal hasn't been loaded. (Esp. because of loading
% prefs). In this case, auto-set the view to anatomy mode, so these 
% functions don't go looking for data that aren't there. (ras, 11/05)
switch v.ui.displayMode
    case 'anat', % do nothing
    case 'map', 
        if isempty(v.map) | isempty(v.map{getCurScan(v)})
            v.ui.displayMode = 'anat';
        end
    case {'co', 'amp', 'ph'}
        if isempty(v.co) | isempty(v.co{getCurScan(v)})
            v.ui.displayMode = 'anat';
        end
end

% Still experimenting with these settings.
prefs = mrmPreferences;
overlayModDepth = prefs.overlayModulationDepth;
dataSmoothIterations = prefs.dataSmoothIterations;
roiDilateIterations = prefs.roiDilateIterations;
clusterThreshold = prefs.clusterThreshold;
if isfield(prefs, 'dimCmap')
    dimCmap = prefs.dimCmap;
else
    dimCmap = 1;
end
if(strcmp(prefs.overlayLayerMapMode,'mean'))
    layerMapMethod = 0;
elseif(strcmp(prefs.overlayLayerMapMode,'max'))
    layerMapMethod = 1;
else
    warning(['Unknow value for mrmPreference overlayLayerMapMode ('...
             prefs.overlayLayerMapMode '). Run mrmPreferences.']);
    layerMapMethod = 0;
end

% The vertex gray map can now be N x M, where N is the the number of data
% 'layers' that map to each of the M vertices. 
vertexGrayMap = meshGet(msh,'vertexGrayMap');

if(strcmp(prefs.layerMapMode,'all'))
    mapToAllLayers = true;
    if(size(vertexGrayMap,1)==1)
        vertexGrayMap = mrmMapVerticesToGray(msh.initVertices, viewGet(v,'nodes'), ...
                                             viewGet(v,'mmPerVox'), viewGet(v,'edges'));
        msh.vertexGrayMap = vertexGrayMap
        v = viewSet(v,'currentmesh',msh);
    end
else
    mapToAllLayers = false;
end


% If transparency is disabled, then we can save a bit of time by not
% dealing with the the alpha channel.
% transparencyFlag = 0;

% Initialize the new color overlay to middle-grey. This reduces 
% edge artifacts when we smooth.
sz = size(meshGet(msh,'colors'),2);

% Center new colors around 127.  Don't forget alpha channel.  These are the
% vertices that are painted in the 3D view.
newColors = ones(4,sz) + 127;

% Map color overlay to vertices
% vertexGrayMap tells us which gray node to use for each vertex.
dataMask = zeros(1,sz);

% VertInds will be a logical map that tells us which vertices have at least
% one data layer that is not null. Some vertices (find(vertInds==0)) have
% no data associated with them. Note that this mask is different from the
% dataMask, which tells us which data values are below the thresholds that
% the user has set. 
vertInds = (vertexGrayMap(1,:) > 0);
if(mapToAllLayers)
    for(ii=2:size(vertexGrayMap,1))
        vertInds = vertInds | (vertexGrayMap(ii,:) > 0);
    end
end
vertInds = logical(vertInds);

% Create the Color Overlay and Data Mask.  The dataMaskIndices are the
% indices into the data (e.g., co{1}(dataMaskIndices) are the entries that
% have valid data values given the thresholds.  The color overlay describes
% the RGB values for ALL of the data, not just the valid points described
% in dataMaskIndices.
%[dataMaskIndices,colorOverlay] = meshCODM(v);
% 09/2005 SOD: added a flag (phaseFlag) which indicates whether the
% data (data) is phase (circular) or not. This will have an impact
% on the kind of smoothing that is done later on.
[dataMaskIndices, data, cmap, dataRange, phaseFlag] = meshCODM(v);

% Big move.  The new colors are assigned from the color overlay.  The color
% overlay was found from the VOLUME data.  The mapping from volume
% locations to vertex indices is managed by vertexGrayMap.  Each vertex is
% mapped to a gray node- except the vertices that have no nearest gray
% node- they are zeroed-out in vertexGrayMap and here are flagged by
% vertInds. Remeber- the indices to vertexGrayMap are in vertex space-
% each value in vertexGrayMap corresponds to an entry in 
% mesh.vertices and specifies which gray matter node is closest to that
% vertex.
%
if(mapToAllLayers & size(vertexGrayMap,1)>1 & ~isempty(data))
    % Initialize dataOverlay with the mean data value to avoid bias when
    % smoothing.
    %dataOverlay = zeros(size(vertexGrayMap(vertInds)));
    allV = vertexGrayMap > 0;
    % ignore non-finite data such as NaN's
    tmpdata = data(vertexGrayMap(allV(:)));
    dataOverlay = repmat(mean(tmpdata(isfinite(tmpdata))),1,sz);
    % dataOverlay = repmat(mean(data(vertexGrayMap(allV(:)))),1,sz);
    
    if(layerMapMethod==0)
      
      if ~phaseFlag,
        n = sum(allV,1);
        for(ii=1:size(vertexGrayMap,1))
          dataOverlay(allV(ii,:)) = dataOverlay(allV(ii,:)) + data(vertexGrayMap(ii,allV(ii,:)));
        end
        dataOverlay(vertInds) = dataOverlay(vertInds)./n(vertInds);
        
      else, % for phase data go complex average and go back
        % recompute complex dataOverlay, taking only finite data
        tmpdata = -exp(i*data(vertexGrayMap(allV(:))));
        dataOverlay = repmat(mean(tmpdata(isfinite(tmpdata))),1,sz);
        %dataOverlay = repmat(mean(-exp(i*data(vertexGrayMap(allV(:))))),1,sz);
        n = sum(allV,1);
        for(ii=1:size(vertexGrayMap,1))
          dataOverlay(allV(ii,:)) = dataOverlay(allV(ii,:)) + -exp(i*data(vertexGrayMap(ii,allV(ii,:))));
        end
        dataOverlay(vertInds) = angle(dataOverlay(vertInds)./n(vertInds))+pi;        
      end;
      
    else
        curData = zeros(size(dataOverlay));
        for(ii=1:size(vertexGrayMap,1))
            curData(:) = -9999;
            curData(allV(ii,:)) = data(vertexGrayMap(ii,allV(ii,:)));
            biggerInds = curData>dataOverlay;
            dataOverlay(biggerInds) = curData(biggerInds);
        end
    end
else
    dataOverlay = zeros(size(vertInds));
    if ~isempty(data)
        dataOverlay(vertInds) = data(vertexGrayMap(1,vertInds));
    end
end

%newColors(1:3,vertInds) = colorOverlay(:,vertexGrayMap(vertInds));

% The dataMaskIndices are indices into the gray matter nodes. vertexGrayMap
% maps each vertex to some gray matter nodes and vertInds just tells us which 
% entries in vertexGrayMap are non-zero (zero entries mean that vertex has 
% no gray node close to it). So, we use ismember to create a new data mask
% that is in vertex space rather than node space. (ie. for each non-zero
% gray-node in vertexGrayMap, is it a memeber of dataMaskIndices?)
tmp = ismember(vertexGrayMap, dataMaskIndices);
dataMask = any(tmp,1);
dataMask(~vertInds) = 0;

% We need the connection matrix to compute clustering and smoothing, below.
conMat = meshGet(msh,'connectionmatrix');
if isempty(conMat) 
    msh = meshSet(msh,'connectionmatrix',1); 
    conMat = meshGet(msh,'connectionmatrix'); 
end

% Remove vertices that have fewer than clusterThreshold neighbors.
if(clusterThreshold>1)
    dataMask = (sum(double(conMat)*double(dataMask'), 2) > clusterThreshold);
end

% SMOOTHING.
if(dataSmoothIterations>0)
    for t=1:dataSmoothIterations
        % Smooth and re-threshold the data mask
        dataMask = connectionBasedSmooth(conMat, double(dataMask));
        dataMask = dataMask>=0.5;
        % New: smooth the data, not the colors (RFD 2005.08.15)
        % 09/2005 SOD: smoothing the data can only be done if the
        % data is non-circular (ie not valid for phase maps). So
        % two ways depending on the data:
        if ~phaseFlag,
          dataOverlay = connectionBasedSmooth(conMat,dataOverlay);
        else,
          % phase data, so go complex, smooth and go back
          dataOverlay = -exp(i*dataOverlay);
          dataOverlay = connectionBasedSmooth(conMat,dataOverlay);
          dataOverlay = angle(dataOverlay)+pi;
        end;
                  
%         for thisPlane=1:3
%             mx = max(newColors(thisPlane,:));
%             mn = min(newColors(thisPlane,:));
%             newColors(thisPlane,:) = connectionBasedSmooth(conMat, newColors(thisPlane,:))';
%             newColors(thisPlane,:) = newColors(thisPlane,:)-min(newColors(thisPlane,:));
%             newColors(thisPlane,:) = newColors(thisPlane,:)./max(newColors(thisPlane,:));
%             newColors(thisPlane,:) = newColors(thisPlane,:).*(mx-mn)+mn;
%         end
    end
end

% compute newColors now after data smoothing
if ~isequal(v.ui.displayMode, 'anat') & sum(data) ~= 0
    % dim color map if that preference is set
    if (dimCmap ~= 1), cmap = dimCmap .* cmap; end
        
    % 09/2005 SOD: remove NaNs from vertInds. This is otherwise done by 
    % meshData2Colors, changing the size of the output colors, which
    % will consequently result in an error due to subscripted
    % assignment dimension mismatch. 
    vertInds(find(isnan(dataOverlay)))=0;
    newColors(1:3,vertInds) = meshData2Colors(dataOverlay(vertInds), cmap, dataRange);
end

% Assign the anatomy colors to the locations where there are no data
% values.
oldColors = meshGet(msh,'colors');
newColors(1:3,~dataMask) = oldColors(1:3,~dataMask);

%
% Overlay ROI colors on top of the data color overlay
%
% NOTE: the following code is slow and should be optimized. Also, we need
% to allow ROIs to be shown as perimeters.
% 
if(abs(v.ui.showROIs)==0)
    roiList = [];
elseif(abs(v.ui.showROIs)==1)
    roiList = v.selectedROI;
else
    roiList = [1:length(v.ROIs)];
end
% roiList can (should) not be 0
if(roiList == 0)
  roiList = [];
end
[stdColVals,stdColLabs]  = meshStandardColors;
% Substitute the ROI color for appropriate nearest neighbor vertices
allRoiVertInds = [];
for iR=roiList
    roi = v.ROIs(iR);
    [junk,nodeInds] = ismember(roi.coords', v.nodes([2,1,3],:)', 'rows');
    nodeInds = nodeInds(nodeInds>0);
    % The following will give us *a* vertex index for each gray node. However,
    % the same gray node may map to multiple vertices, and we want them
    % all. (Otherwise, the color overlay will have holes.) So, we loop
    % until we've got them all.
    [junk,roiVertInds] = ismember(nodeInds, vertexGrayMap(1,:));
    roiVertInds = roiVertInds(roiVertInds>0);
    while(any(junk))
        vertexGrayMap(1,roiVertInds) = 0;
        [junk,tmp] = ismember(nodeInds, vertexGrayMap(1,:));
        roiVertInds = [roiVertInds; tmp(tmp>0)];
    end
    
    % Sometimes we want to dilate the ROIs a bit. The following does this
    % by finding all the neighboring vertices for all the ROI vertices (now
    % stored in roiVertInds). The sparse connection matrix comes in handy 
    % once again, making this a very fast operation.
    if(roiPerimThickness)
        % Just do the perimeter stuff
        origROI=roiVertInds;
        for  t=1:roiPerimThickness
            neighbors = conMat(:,roiVertInds);
            [roiVertInds,junk] = find([neighbors]);
            roiVertInds=unique(roiVertInds);
            
        end
        % Subtract the original from the dilated version. 
        roiVertInds=setdiff(roiVertInds,origROI);        
    else
        % We can dilate the area if we are not also rendering the perimeter...
        for t=1:roiDilateIterations
            neighbors = conMat(:,roiVertInds);
            [roiVertInds,junk] = find(neighbors);
            roiVertInds=unique(roiVertInds);
        end
        
    end
    % *** TO DO: Pull this ROI mask code ou and do it separately, so that
    % any masking that occurs does interfere with the drawing of other
    % (non-mask) ROIs. (RFD)
    if(strcmp(roi.name,'mask'))
        % Don't show the ROI- just use it to mask the data.
        tmp = logical(ones(size(dataMask)));
        tmp(roiVertInds) = 0;
        newColors(1:3,tmp) = oldColors(1:3,tmp);
        clear tmp;
    else     
        if (ischar(roi.color))
            colVal = stdColVals(:, findstr(stdColLabs, roi.color));
        else
            % Check to see if we have a 3x1 vector. 
         colVal = roi.color(:);
            if (length(colVal)~=3)
                error('Bad ROI color');
            end
        end
       newColors(1:3,roiVertInds) = repmat(colVal*255,1,length(roiVertInds));
       allRoiVertInds = [allRoiVertInds; roiVertInds];
    end
end
newColors(newColors>255) = 255;
newColors(newColors<0) = 0;

% Brian asks: Clueless.  What is going on here?
%
% This is an option that will maipulate the overlay colors- adjusting them
% to allow the underlying mesh colors to come through. (Ie. simulate
% transparency). This is useful when your mesh is completely painted, but
% you need some clue about the surface curvature (which is presumable what
% the original mesh colors represent). -Bob
if(overlayModDepth>0)
    % The data mask does not include the vertices that have a ROI color but
    % no data overlay. We want to modulate everything, including ROIs, so
    % we add the roiVertInds to the mask.
    dataMask(allRoiVertInds) = 1;
    newColors(:,dataMask) = (1-overlayModDepth)*newColors(:,dataMask) ...
        + overlayModDepth*double(oldColors(:,dataMask));
    newColors(newColors>255) = 255;
    newColors(newColors<0) = 0;
end

% Place the new colors in the rendering
newColors = uint8(round(newColors));

% Sometimes we just want the values.  Usually, we want to show the data.
if showData
    mrmSet(msh,'colors',newColors');
end

return;
