% mrInitRet
%
% Time-series extraction and analysis script.
% Starts with recon'd P*.mag files and corresponding
% inplane-anatomy images, and creates time series
% files within a designated image-crop range and time period.
% Creates mrSESSION and dataTYPES data structures.
%
% Does the following step:
% - crop inplanes
% - build mrSESSION (& modify if necessary?)
% - build dataTYPES
% - modify analysis parameters in dataTYPES 
% - create Readme
% - extract time series from mag files
% - corAnal
%
% DBR, 3/99  Rewritten based on an earlier version by GB, and
% much hacked subsequently.
% Last modified $Date: 2006/03/03 22:26:39 $
clear all
mrGlobals

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Opening dialogs

initOptions = {'Setup Raw/ Directory',...
        'Crop inplane images',...
        'Create/edit data structures',...
        'Extract time series',...
        'Perform blocked analysis'};

initReply = buttondlg('mrInitRet', initOptions);
if length(find(initReply)) == 0, return; end
doRawDir = initReply(1); 
doCrop = initReply(2)
doSession = initReply(3);
doTSeries = initReply(4);
doCorrel = initReply(5);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Set up the Raw/ subdirectory if selected 
if doRawDir==1, SetupRawDir; end

% Get a valid set of directories:
dirS = GetDirectory(pwd);
if length(dirS) == 0
    disp('Aborted'); 
    return 
end
HOMEDIR = dirS.home;
rawDir = dirS.raw;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get the inplane anatomies and the Pfile header info

% Get the scan params from the Pfile headers
scanParams = GetScanParams(rawDir);


% Load the inplane-anatomy images and initialize the inplanes structure
[anat, inplanes, doCrop] = InitAnatomy(HOMEDIR, rawDir, doCrop);
if isempty(anat)
    disp('Aborted')
    return
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Do crop

% At this point we should have a valid inplane anatomy, and the
% doCrop flag indicates if it can and should be cropped
if doCrop
    % Sometimes inplanes.FOV is not an integer...
    if inplanes.FOV ~= round(inplanes.FOV);
        disp(['Original inplane FOV was ',num2str(inplanes.FOV),' mm, now rounded to ',int2str(round(inplanes.FOV)),' mm.']);
        inplanes.FOV = round(inplanes.FOV);
    end

    
    % Calculate the cropRatio from inplanes.fullSize and functionals(:).fullSize
    cropRatio = 1;
    for scan = 1:length(scanParams)
        cropRatio = max([cropRatio, inplanes.fullSize ./ scanParams(scan).fullSize]);
    end
    
    % Crop the inplane anatomy if requested or not previously done:
    [anat, inplanes] = CropInplanes(rawDir, anat, inplanes, cropRatio);
    if isempty(anat)
        disp('Crop inplanes aborted');
        return
    end
    
    % Delete tSeries (if there are any); they are out of date because the crop has changed
    datadir = fullfile(HOMEDIR,'Inplane','Original','TSeries');
    [nscans,scanDirList] = countDirs(fullfile(datadir,'Scan*'));
    if nscans > 0
        deleteFlag = questdlg(['The existing tSeries are out of date because the crop has changed. Delete existing TSeries?'],...
            'Delete tSeries','Yes','No','Yes');
        if strcmp(deleteFlag,'Yes')
            for s=1:nscans
                delete(fullfile(datadir,scanDirList{s},'*.mat'));
            end
        end
    end
    
else
    % Check that previous crop information is present:
    if ~isfield(inplanes, 'crop')
        % Whoops -- no previous crop info. Try to get from anat.mat file:
        Alert('Problems with inplane crop');
        return
    end
end

% Save anat
anatFile = fullfile(HOMEDIR, 'Inplane', 'anat.mat');
save(anatFile, 'anat', 'inplanes');


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Create/load mrSESSION and dataTYPES, modify them, and save

% If mrSESSION already exits, load it.
sessionFile = fullfile(HOMEDIR, 'mrSESSION.mat');
if exist(sessionFile, 'file')
    loadSession;
    % if docrop, make sure that the mrSESSION is up-to-date
    if doCrop
        mrSESSION.inplanes = inplanes;
        mrSESSION = UpdateSessionFunctionals(mrSESSION,scanParams);
        saveSession;
    end
end

if doSession
    % If we don't yet have a session structure, make a new one.
    if isempty(mrSESSION)
        mrSESSION = CreateNewSession(HOMEDIR, inplanes, mrLoadRetVERSION);
    end
    
    % Update mrSESSION.functionals with scanParams corresponding to any new Pfiles.
    % Set mrSESSION.functionals(:).crop & cropSize fields
    mrSESSION = UpdateSessionFunctionals(mrSESSION,scanParams);
    
    % Dialog for editing mrSESSION params:
    [mrSESSION,ok] = EditSession(mrSESSION);
    if ~ok
        disp('Aborted'); 
        return
    end
    
    % Create/edit dataTYPES
    if isempty(dataTYPES)
        dataTYPES = CreateNewDataTypes(mrSESSION);
    else
        dataTYPES = UpdateDataTypes(dataTYPES,mrSESSION);
    end
    dataTYPES(1) = EditDataType(dataTYPES(1));
    
    % Save any changes that may have been made to mrSESSION & dataTYPES
    saveSession;
    
    % Create Readme.txt file
    mrCreateReadme;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Extract time series & perform corAnal

% Create time series files
if doTSeries
    endian = LittleEndianCheck(mrSESSION);
    GetRecon(rawDir,0,0,[],endian); 
end

% Perform correlation analysis
if doCorrel
    computeCorAnal(initHiddenInplane, 0, 1);
    computeMeanMap(initHiddenInplane, 0, 1);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Clean up

clear all

return



% DISABLED: Scale FOV; don't think anyone usees this anymore...
%     %%%%%%%%%%%%%% Do FOV scaling before crop
%     fovchoice = questdlg({'Scale FOV of Inplane Anatomy?'; '(Choose NO if unknown -- or ask Junjie)'},'scale anatomy','Yes','No','No');
%     switch fovchoice;
%         case 'Yes'
%             newfov = [];  % must input a number in the loop below
%             while isempty(newfov);
%                 newfov = inputdlg(['Original FOV is ',num2str(inplanes.FOV),' mm, Please input NEW desired FOV (mm):'],'Please input FOV');
%                 if iscell(newfov)&~isempty(newfov); newfov = str2num(newfov{1}); end;
%             end
%             fprintf('Interpolating anatomy (takes long time)...');
%             fovRatio = [round(newfov), inplanes.FOV]/gcd(round(newfov),inplanes.FOV);
%             [anat, inplanes] = ReduceFOV(anat, inplanes, fovRatio);
%             fprintf('finished!');
%     end