function gray = createCombinedSession(newSessDir, srcSessions);

% gray = createCombinedSession([newSessDir], [srcSessions], []);
%
% Create a new mrVista session for the purposes of combining
% data across sessions. This is only for sessions which 
% have gray tSeries, and are from the same subject/segmentation.
%
% This session will be distinct from other sessions, in that
% it will only have a gray view, not an inplane view (diff't 
% sessions have different inplane Rxs -- it may be possible to 
% create an interpolated inplane set but that's way advanced).
% The Gray coords.mat file will contain a union of the coordinates
% from all of the source sessions. In the related functions (see
% importTSeries), any coordinates which lie outside a given
% session's prescription will be assigned NaN.
%
% newSessDir is a path to the new directory to create.
% If it doesn't exist, creates the directory. Creates
% a mrSESSION file for the combined session. If some 
% fields in the mrSESSION and dataTYPES variables are
% ambiguous, it will leave them blank or else grab
% them from the first source session. If omitted, 
% newSessDir is taken to be the current dir.
%
% srcSessions is a cell of strings, containing paths
% to each of the sessions that will contribute to the 
% combined coordinates. If omitted, pops up a prompt.
% (The prompt initially looks at dirs in the parent
% directory, but you can navigate to other dirs by
% selecting the (Find Other Session) option.)
%
% The main effect of the source sessions is to contribute
% relevant gray coordinates. If you create a combined
% session, but later want to add data from another session,
% you may be ok. If the new session doesn't cover any
% new gray matter not covered by the source sessions, 
% you can just add the data. Otherwise, you'll need to 
% rebuild the combined session and import the data all
% over again.
%
% hemispheres is an optional flag to specify the hemispheres
% from which to take the segmentation info. By default it
% is 'both', take from both hemispheres, but if you specify
% 'right' or 'left' it will take only coords from the appropriate
% hemisphere.
% 
% Note that this function doesn't take any of the
% scan data yet; it just initializes the session so
% you can add scans to it.
%
%
% ras 03/05
if notDefined('newSessDir'),    newSessDir = pwd;        end

callingDir = pwd;

if ieNotDefined('srcSessions')
    % create an interface, get sessions
	studyDir = fileparts(pwd); % guess at study dir
    srcSessions = selectSessions(studyDir);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Create new session dir if it doesn't exist %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[p d] = fileparts(fullpath(newSessDir));
if ~exist(newSessDir,'dir')
    mkdir(p,d);
    fprintf('Made directory %s in parent dir %s.\n',d,p);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Check that each source directory exists, and get full paths %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
newSessDir = fullpath(newSessDir);
for i = 1:length(srcSessions)
    if exist(srcSessions{i}, 'dir')
        srcSessions{i} = fullpath(srcSessions{i});
    else
        errmsg = sprintf('Source dir %s not found.', srcSessions{i});
        error(errmsg);
    end
end


%%%%%%%%%%%%%%%%%%%%%%%%
% create mrSESION file %
%%%%%%%%%%%%%%%%%%%%%%%%
cd(newSessDir);
mrGlobals;
mrSESSION = [];
dataTYPES = [];
vANATOMYPATH = '';
tmp = load(fullfile(srcSessions{1},'mrSESSION.mat'));

mrSESSION.mrLoadRetVersion = 3.01;
mrSESSION.sessionCode = d;
desc = 'Combined: coords from ';
for i = 1:length(srcSessions)
    [p f] = fileparts(srcSessions{i});
    desc = [desc f ' '];
end
mrSESSION.description = desc;
mrSESSION.subject = tmp.mrSESSION.subject;
mrSESSION.examNum = 'N/A';
mrSESSION.inplanes = tmp.mrSESSION.inplanes;
mrSESSION.functionals = [];
mrSESSION.alignment = [];

dataTYPES.name = 'Imported_Original';

% Dilemna: we need various data types fields 
% as placeholders to run mrVista, but don't want
% to actually import any tSeries yet (can't have
% a struct of size 0, it seems). 
% Solution (Hack): Create a dummy, and label it so:
dataTYPES.scanParams = tmp.dataTYPES(1).scanParams(1);
dataTYPES.blockedAnalysisParams = tmp.dataTYPES(1).blockedAnalysisParams(1);
dataTYPES.eventAnalysisParams = tmp.dataTYPES(1).eventAnalysisParams(1);
dataTYPES.scanParams.annotation = '(Empty Scan)';

if isfield(tmp, 'vANATOMYPATH')
    vANATOMYPATH = tmp.vANATOMYPATH;
else
    vANATOMYPATH = getVAnatomyPath(mrSESSION.subject);
end

HOMEDIR = pwd;

mrSessFile = fullfile(newSessDir,'mrSESSION.mat');
if exist(mrSessFile,'file')
    % confirm to save over
    msg = sprintf('%s already exists. Save over?',mrSessFile);
    response = questdlg(msg,'Confirm','Yes','No','No');
    if isequal(response,'No')   
        disp('Aborted createCombinedSession w/o saving.')
        return
    end
end
save(mrSessFile,'mrSESSION','dataTYPES','vANATOMYPATH');
fprintf('Saved new mrSESSSION.mat file.\n');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% create Gray dir, get coords %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ensure a gray dir exists
grayDir = fullfile(HOMEDIR,'Gray');
if ~exist(grayDir)
    mkdir(HOMEDIR,'Gray');
    fprintf('Created Directory: %s \n',fullfile(HOMEDIR,'Gray'));
end

% get coords from each of the source sessions
for i = 1:length(srcSessions)
    srcCoordsFile = fullfile(srcSessions{i},'Gray','coords.mat');
    
    if i==1
        % use 1st session gray coords to set those fields
        % which I'm not sure of:
        load(srcCoordsFile);
    end
    
    srcData{i} = load(srcCoordsFile);
    allCoords{i} = srcData{i}.coords;
end

% find union of all coords
% (a bit tricky, since we want to keep 
% the corresponding nodes at the same time)
coords = [];
nodes = []; 
for i = 1:length(srcSessions)
    coords = [coords allCoords{i}];
    nodes = [nodes srcData{i}.nodes];    
end

% (this will remove redundant coords, and 
% provide an index to use to grab the nodes)
[coords indices] = intersectCols(coords,coords);
nodes = nodes(:,indices);

% save (the other variables were loaded from 1st src file)
coordsFile = fullfile(grayDir,'coords.mat');
if exist(coordsFile,'file')
    % confirm to save over
    msg = sprintf('%s already exists. Save over?',coordsFile);
    response = questdlg(msg,'Confirm','Yes','No','No');
    if isequal(response,'No')   
        disp('Aborted createCombinedSession w/o saving.')
        return
    end
end
save(coordsFile, 'coords', 'nodes', 'edges', 'allLeftNodes', ...
    'allLeftEdges', 'allRightNodes', 'allRightEdges',...
    'leftPath', 'rightPath', 'keepLeft', 'keepRight');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% make a Readme.txt file letting people know    %
% this is a combined session.                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fid = fopen('Readme.txt', 'w');
if fid > -1
    fprintf(fid, 'This is a mrVista Combined Session, intended \n');
    fprintf(fid, 'to hold data from multiple sessions. \n');
    fprintf(fid, 'It has gray coords that span at least \n');
    fprintf(fid, 'the following sessions: \n');
    for i = 1:length(srcSessions)
        fprintf(fid, '%s \n', srcSessions{i});
    end
    fprintf(fid, '\nPlease use only Volume / Gray / Flat views \n');
    fprintf(fid, 'for this session; Inplanes aren''t defined. \n');
    fprintf(fid, '\nSee createCombinedSession and importTSeries');
    fprintf(fid, 'for more info.\n');
    fclose(fid);
end

%%%%%%%%%%%%%%%%%%%%%%
% That should do it. %
%%%%%%%%%%%%%%%%%%%%%%
if nargout>0, gray = initHiddenGray;    end
cd(callingDir);
fprintf('Finished building %s directory.\n',newSessDir);


return

    
    
