function view = volume3View(view, loc, xHairs);
% view = volume3View(view, <loc>, <xHairs>);
%
% Open/refresh a volume '3-view': a 'BrainVoyager'-like
% 3-window view, showing sagittal, coronal, and axial slices, at the
% specified location. 'loc' should be a 3-vector containing, respectiveley,
% the Superior/Inferior, Anterior/Posterior, and Right/Left coordinates.
% (For most of the volume anatomies used, S/I range from 1-256, A/P and
% R/L ranges from 1-124.
%
% If omitted, 'loc' is taken from the view.loc field, or
% failing this set to the middle of the volume.
%
% xHairs: if set to nonzero, will place crosshairs on the current location,
% otherwise will omit them (0 default).
%
% 3/06/03 by ras
% 12/03 ras: using imagesc instead of imshow for now to display
% views, since there's a trickyness in accurately detecting the
% clicked-on location of images put up using imshow.
% 01/04 ras: cleaned up a lot, added 'zoom' field to ui struct allowing
% zooming in on vAnat. (see also zoom3view)
% to do: re-merge ras_recomputeImage with the regular recomputeImage.
% 11/04 ras: added flipping axes option
tic

global HOMEDIR;

if ~exist('view', 'var') | isempty(view)
    view = getSelectedVolume;
end

if ~exist('loc','var')| length(loc)<3
    loc = getLocFromUI(view); 
else
    setLocInUI(view, loc); 
end
view.loc = loc;

% get the ui struct from the view
ui = viewGet(view,'ui');

if ~exist('xHairs','var')
    if isfield(ui, 'crosshairs') & ui.crosshairs==1
        xHairs = 1;
    else
        xHairs = 0;
    end
end

%%%%% check if window exists
winTag = ['3VolumeWindow: ',view.name];
winExists = findobj('Tag',winTag);

% if window doesn't exist, open the window
if isempty(winExists)
    openRaw3ViewWindow;
else
    % if one or more 3-view figures exist, make the most recent one current
    figure(winExists(end));
end

% fprintf('1, %3.2f\n', toc)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ***** Refresh Window ***** %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
modeStr=['ui.',ui.displayMode,'Mode'];
mode = eval(modeStr);
cmap = mode.cmap;

% get images, apply phase/amp/param maps
[view axi] = recompute3ViewImage(view, loc(1), 1);
[view cor] = recompute3ViewImage(view, loc(2), 2);
[view sag] = recompute3ViewImage(view, loc(3), 3);

% fprintf('2, %3.2f\n', toc)


% flip images if selected
if isfield(ui,'flipAP') & ui.flipAP==1
    sag = fliplr(sag);
    % update loc
end
if isfield(ui,'flipLR') & ui.flipLR==1
    axi = fliplr(axi);
    cor = fliplr(cor);
    % update loc
end

% fprintf('3, %3.2f\n', toc)

%%%%%%%%%%%%%%%%%%%%%
% put up each image %
%%%%%%%%%%%%%%%%%%%%%
axes(ui.axiAxesHandle); cla;
image(axi); colormap(cmap);
axis image, axis off
% imshow(axi,cmap);

axes(ui.corAxesHandle); cla;
image(cor); colormap(cmap);
axis image, axis off
% imshow(cor,cmap);

axes(ui.sagAxesHandle); cla;
image(sag); colormap(cmap);
axis image, axis off
% imshow(sag,cmap);

% fprintf('4, %3.2f\n', toc)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%              draw ROIs if selected            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if isfield(view,'ui') & ui.showROIs ~= 0 & length(view.ROIs) > 0
    if abs(ui.showROIs)==1, roiList = view.selectedROI; % show selected ROI only
    else,                   roiList = 1:length(view.ROIs); % show all ROIs
    end

    drawROIs3View(view, roiList, ui.axiAxesHandle, 1, loc);
    drawROIs3View(view, roiList, ui.corAxesHandle, 2, loc);
    drawROIs3View(view, roiList, ui.sagAxesHandle, 3, loc);
end

% fprintf('5, %3.2f\n', toc)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% zoom in on selected coords %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isfield(ui,'zoom')
    % initialize to full range if not set
    ui.zoom = [0 size(view.anat,1);...
        0 size(view.anat,2);...
        0 size(view.anat,3)];
else
    axis(ui.axiAxesHandle, [ui.zoom(3,:) ui.zoom(2,:)]);
    axis(ui.corAxesHandle, [ui.zoom(3,:) ui.zoom(1,:)]);
    axis(ui.sagAxesHandle, [ui.zoom(2,:) ui.zoom(1,:)]);
end

% fprintf('6, %3.2f\n', toc)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% add crosshairs to each image to specify current location %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if xHairs
    view = renderCrosshairs(view, 1);
end


% set a callback for each image which dynamically updates
% the location based on where the user clicks
% callback:
% view = recenter3view(view,[orientationFlag]);
axes(ui.axiAxesHandle);
cb = [view.name,'=recenter3view(',view.name,', 1);'];
set(get(gca,'Children'), 'ButtonDownFcn', cb);
axes(ui.corAxesHandle);
cb = [view.name,'=recenter3view(',view.name,', 2);'];
set(get(gca,'Children'), 'ButtonDownFcn', cb);
axes(ui.sagAxesHandle);
cb = [view.name,'=recenter3view(',view.name,', 3);'];
set(get(gca,'Children'), 'ButtonDownFcn', cb);

% fprintf('7, %3.2f\n', toc)


% Add titles to indicate directions (esp. L-R: radiological conventions)
txtcol = 'k';
SIsz = ui.zoom(1,2) - ui.zoom(1,1);
APsz = ui.zoom(2,2) - ui.zoom(2,1);
LRsz = ui.zoom(3,2) - ui.zoom(3,1);
minS = ui.zoom(1,1);
minA = ui.zoom(2,1);
minL = ui.zoom(3,1);
axes(ui.sagAxesHandle);
if isfield(ui,'flipAP') & ui.flipAP==1
    title('Pos \leftrightarrow Ant','FontSize',10,'Color',txtcol);
else
    title('Ant \leftrightarrow Pos','FontSize',10,'Color',txtcol);
end
axes(ui.corAxesHandle);
if isfield(ui,'flipLR') & ui.flipLR==1
    title('Right \leftrightarrow Left','FontSize',10,'Color',txtcol);
else
    title('Left \leftrightarrow Right','FontSize',10,'Color',txtcol);
end

% the I/S and P/A directions for the cor, axial don't
% get flipped...
text(minL-0.2*LRsz,minS + 0.5*SIsz, 'Inf \leftrightarrow Sup', ...
    'FontSize', 10,  'HorizontalAlignment', 'center', ...
    'Rotation', 90, 'Color', txtcol);
axes(ui.axiAxesHandle);
text(minL-0.2*LRsz,minA + 0.5*APsz, 'Pos \leftrightarrow Ant', ...
    'FontSize', 10,  'HorizontalAlignment', 'center', ...
    'Rotation', 90, 'Color', txtcol);

% set tags so we can find each of these axes later
set(ui.sagAxesHandle, 'Tag', 'sagAxes');
set(ui.corAxesHandle, 'Tag', 'corAxes');
set(ui.axiAxesHandle, 'Tag', 'axiAxes');

% fprintf('8, %3.2f\n', toc)

% % selecting the axials by default is nice
% axes(ui.axiAxesHandle);

% if slice number fields are set up for manually paging
% through slices, update these as well
if isfield(view,'ui') & isfield(ui, 'sliceNumFields')
    set(ui.sliceNumFields(1), 'String', num2str(loc(1)));
    set(ui.sliceNumFields(2), 'String', num2str(loc(2)));
    set(ui.sliceNumFields(3), 'String', num2str(loc(3)));
end

% Draw colorbar
str = [ui.displayMode,'Mode'];
modeStr=['ui.',str];
mode = eval(modeStr);
cmap = mode.cmap;
numGrays = mode.numGrays;
numColors = mode.numColors;
clipMode = mode.clipMode;
numGrays = mode.numGrays;
numColors = mode.numColors;
if isempty(view.ui.cbarRange)
    setColorBar(view, 'off', [numGrays+1 numGrays+numColors]);
else
    setColorBar(view, 'on', [numGrays+1 numGrays+numColors]);
end

% lastly, update the annotation text
set(ui.annotationHandle, 'String', annotation(view, getCurScan(view)));

% fprintf('9, %3.2f\n\n', toc)


return
% /----------------------------------------------------------------/ %





% /----------------------------------------------------------------/ %
function loc = getLocFromUI(view);
% reads off the axi,cor, and sag UI edit fields to
% get the current view location.
loc(1) = str2num(get(view.ui.sliceNumFields(1),'String'));
loc(2) = str2num(get(view.ui.sliceNumFields(2),'String'));
loc(3) = str2num(get(view.ui.sliceNumFields(3),'String'));
return
% /----------------------------------------------------------------/ %





% /----------------------------------------------------------------/ %
function view = setLocInUI(view, loc);
% reads off the axi,cor, and sag UI edit fields to
% get the current view location.
set(view.ui.sliceNumFields(1), 'String', num2str(loc(1)));
set(view.ui.sliceNumFields(2), 'String', num2str(loc(2)));
set(view.ui.sliceNumFields(3), 'String', num2str(loc(3)));
return
% /----------------------------------------------------------------/ %




% /----------------------------------------------------------------/ %
function drawROIs3View(view, roiList, axs, ori, loc);
% draw the ROIs, parsing the relevant settings, on the axes 
% specified by [axs]. This is implemented a little differently
% from drawROIs or drawROIsPerim, in that it doesn't select a given
% orientation each time (which caused some extra time during refreshes).
% roiList is an array into the view.ROIs field.
ui = view.ui;

% build prefs for the outline function
prefs.method = 1+(ui.showROIs>0); % 1=perimeter, 2=boxes over each pixel
prefs.axesHandle = axs;

for r = roiList
    R = view.ROIs(r);
    if ~isempty(R.coords)
        prefs.color = R.color;
        if r==view.selectedROI, prefs.color=viewGet(view,'selRoiColor'); end

        switch ori
            case 1, pts = R.coords([2 3], R.coords(1,:)==loc(1));
            case 2, pts = R.coords([1 3], R.coords(2,:)==loc(2));
            case 3, pts = R.coords([1 2], R.coords(3,:)==loc(3));
        end

        if isfield(ui, 'flipLR') & ui.flipLR==1 & ori<3
            % L/R flip affects columns of axi + coronal, but not sag, orientations
            dims = viewSize(view);
            pts(2,:) = dims(2) - pts(2,:);
        end


        outline(pts, prefs);
    end
end

return
