function view = refreshView(view,recomputeFlag)
% 
% view = refreshView(view,recomputeFlag)
% 
% Author: Who Knows
% Purpose:
% Refresh the screen given the current state of the ui
% parameters.  This is a key script.  It is used to put up all
% the different viewTypes of images under the View menu.
%
% view.ui.displayMode must be a string (either 'anat', 'co',
% 'amp', 'ph, or 'map') corresponding to the available
% displayModes (anatMode, coMode, ampMode, phMode, and mapMode).
% Each displays a different overlay on the anatomy, with a
% different colormap.
%
% Each displayMode has a cmap field (e.g., view.ui.anatMode.cmap).
%
% Each displayMode also has a clipMode field (e.g.,
% view.ui.anaMode.clipMode) that determines clipping of the
% overlay image values to fill the range of available colors.  The
% clipMode can be either 'auto' or [clipMin clipMax].  If 'auto',
% uses overlayClip sliders to determine clipping.  If [clipMin
% clipMax], uses those values instead.  For some of the
% displayModes ('ph') fixed clipping usually makes sense because
% we want certain values (e.g., 0 phase) to always correspond to
% the same color (e.g., red) regardless of the cothresh and
% phWindow.
%
% recomputeFlag: if nonzero (the default), recomputes the uint8
% image.  Otherwise, uses the image stored in view.ui.image.
% 
% djh, 1/97
% rmk, 1/15/99 added title bar updates for map views
% rfd, 3/31/99 added option for drawing ROIs as outlines
%              (a negative view.ui.showROIs flags this).
% wap, 12/22/99
%	 Removed 'Refresh... view' print statement.  It was getting
%   annoying that information would wrap off the screen.
% bw, 12/29/00
%   added support for scan slider redisplay.
% djh, 2/22/2001 updated to version 3


% TODO
% We ned to make sure that the VOLUME values (e.g., current roi) are
% properly set in the ui fields.  This is not currently enforced here (BW).
if ~exist('recomputeFlag','var'), recomputeFlag=1; end

% Make this one the selectedInplane, selectedVolume, or selectedFlat
if iscell(view), selectView(view); end

% Set window title
% Check for sessionCode - this should have been set in openxxxxWindow but
% some other programs (like mrAlign) don't beshave properly.
if (~isfield(view,'sessionCode'))
    view.sessionCode=pwd;
end

% Get colormap, numGrays, numColors and clipMode
str = [view.ui.displayMode,'Mode'];

% Make sure the phMode, or ampMode structure exists.  If not, then set them
% to the default values.
if ~checkfields(view,'ui',str), view=resetDisplayModes(view); end

% Here is the mode, get the relevant parameters.
modeStr=['view.ui.',str];
mode = eval(modeStr);
cmap = mode.cmap;
numGrays = mode.numGrays;
numColors = mode.numColors;
clipMode = mode.clipMode;

if (recomputeFlag | isempty(view.ui.image))
    view = recomputeImage(view,numGrays,numColors,clipMode);
end

% Select the window
set(0,'CurrentFigure',view.ui.windowHandle);

% Update annotation string
set(view.ui.annotationHandle,'string',annotation(view,getCurScan(view)));

% Display final image
% Change this to cope with different rotations in L and R flatmaps. For
% VOLUME, INPLANE views this does nothing.
rotateDeg=0;

if (strcmp(view.viewType,'Flat'))
    [rotations,flipLR]=getFlatRotations(view);
    
    if (~(recomputeFlag==2)) % Only set when the hemisphere changes
        
        % Retreive the imageRotation from the UI
        thisRotation=getImageRotate(view);
        
        % Set the corresponding value in degrees in FLAT.imageRotateDegrees
        rotations(getCurSlice(view))=thisRotation*(180/pi);
    end
    
    view=setFlatRotations(view,rotations,flipLR);
    rotateDeg=rotations(getCurSlice(view));
    flipFlag=flipLR(getCurSlice(view));
    
    if (rotateDeg | flipFlag) %If this has been set, then do a rotation
        
        im=ind2rgb(view.ui.image,cmap);
        
        %         if (flipFlag)           
        %             im=imrotate(im,rotateDeg,'bicubic','crop');
        %         else
        im=imrotate(im,-rotateDeg,'bicubic','crop');           
        %         end % end if flipFlag
        
        [im,cmap]=rgb2ind(im,cmap);
        
        if (flipFlag), im=fliplr(im); end % if flipflag
        image(im);
        colormap(cmap);
        axis image;
        axis off;
    else
        image(view.ui.image);
        colormap(cmap);
        axis image;
        axis off;
    end
else
    image(view.ui.image);
    colormap(cmap);
    axis image;
    axis off;
end


% Draw colorbar
if isempty(view.ui.cbarRange)
    setColorBar(view,'off',[numGrays+1,numGrays+numColors]);
else
    setColorBar(view,'on',[numGrays+1,numGrays+numColors]);
end


% Draw ROIs
if view.ui.showROIs > 0,     drawROIs(view);
elseif view.ui.showROIs < 0, drawROIsPerimeter(view);
end

% also added zoom capability:
if isfield(view.ui,'zoom') 
    if ismember(view.viewType, {'Inplane' 'Flat' 'SS'}) 
        try, axis(view.ui.zoom); end
    elseif ismember(view.viewType, {'Volume' 'Gray'})
        % 3-D zoom bounds specified:
        z = view.ui.zoom;
        switch getCurSliceOri(view)
            case 1, axis([z(3,1) z(3,2) z(2,1) z(2,2)]); % axi
            case 2, axis([z(3,1) z(3,2) z(1,1) z(1,2)]); % cor
            case 3, axis([z(2,1) z(2,2) z(1,1) z(1,2)]); % sag
        end
    end
end

% add a fancy GUI callback
htmp = get(gca,'Children');
set(htmp,'ButtonDownFcn',sprintf('%s = adjustMRImage(%s);',view.name,view.name));

% if isfield(view, 'mode3d')
%     if view.mode3d == 2, view = Refresh3D(view); end
% end

return;

