function [montage,view] = inplaneMontage(view,slices,mode,zoom);
% [montage,view] = inplaneMontage(view,[slices],[mode],[zoom]):
%
% display a montage of all the inplane slices in the Inplane view window,
% with the selected mode overlayed.
%
% 'slices:' vector specifying which slices to show. If omitted, shows all
% slices.
%
% 'mode', if not entered, defaults to the currently selected view mode.
%
% For hidden views, mode can be entered as a struct, with info that
% would otherwise be gotten from the UI:
%   mode.clipMode -- 'auto' or [min max] of clipping for any overlay
%   mode.numGrays -- # gray values for underlay/anat
%   mode.numColors -- # colors for overlay
%   mode.cmap -- a color map
%   mode.cothresh -- coherence threshold for overlay
%   mode.mapWindow -- window of parm map vals for overlay
%   mode.phWindow -- phase window for overlay
%   mode.displayMode -- name of display mode to use
%   These fields aren't all required, and reasonable defaults will
%   be selected if any are omitted.
%
% 'zoom', if entered, will only make a montage of the selected zoomed
% region. zoom should be a 1x4 vector of values in the format of axis
% values [xmin xmax ymin ymax]
%
% 02/20/04 ras. (a lot of this is taken from recomputeImage)
% 09/04 ras -- changed name to inplaneMontage, to be consistent
% with the term used elsewhere (OED seems to suggest this is also
% a better description).
% 06/05 ras -- added ability to pass in mode struct for hidden views.
% 03/06 ras -- for auto clip modes, now sets clip range to be min/max of 
%              all rendered slices, not just the last one.
if ~exist('slices','var') | isempty(slices),
    slices = [1:numSlices(view)];
end

if ~exist('mode','var') | isempty(mode),
    mode = view.ui.displayMode;
end

if ~exist('zoom','var') | isempty(zoom),
    % TO DO: upate viewGet to get zoom properly for
    % all view types
    if checkfields(view, 'ui', 'zoom');
        zoom = view.ui.zoom;
    else
        dims = viewSize(view);
        zoom = [1 dims(2) 1 dims(1)];
    end
end

% Initialize images
montage = [];
anatIm = [];
overlay = [];

if isstruct(mode)
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % for hidden views, get mode info from struct %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if ~isfield(mode,'clipMode'), mode.clipMode = 'auto'; end
    if ~isfield(mode,'numGrays'), mode.numGrays = 128; end
    if ~isfield(mode,'numColors'), mode.numColors = 128; end
    if ~isfield(mode,'cmap'), mode.cmap = hotCmap(128,128); end

    if ~isfield(mode,'cothresh'), mode.cothresh = 0; end
    if ~isfield(mode,'mapWindow'), mode.mapWindow = [0 1000]; end
    if ~isfield(mode,'phWindow'), mode.phWindow = [0 2*pi]; end

    if ~isfield(mode,'displayMode'), mode.displayMode = 'anat'; end

    cmap = mode.cmap;
    numColors = mode.numColors;
    numGrays = mode.numGrays;
    clipMode = mode.clipMode;
    cothresh = mode.cothresh;
    mapWindow = mode.mapWindow;
    phWindow = mode.phWindow;

    mode = mode.displayMode;
else
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % non-hidden views: get info from UI %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    modeInfo = viewGet(view,[mode 'Mode']);

    % Get cothresh, phWindow, and mapWindow from sliders
    cothresh = getCothresh(view);
    phWindow = getPhWindow(view);
    mapWindow = getMapWindow(view);
    clipMode = modeInfo.clipMode;
    numGrays = modeInfo.numGrays;
    numColors = modeInfo.numColors;
    cmap = modeInfo.cmap;
end


% initialize clip values for overlay
clipMin = 0;
clipMax = 0.1;

% get # of slices; figure out a good # of rows, cols for montage
nSlices = length(slices);
nrows = ceil(sqrt(nSlices));
ncols = ceil(nSlices/nrows);

for row = 1:nrows
    rowIm = [];

    for col = 1:ncols
        sliceind = (row-1)*ncols + col;

        if sliceind <= length(slices)
            % if there's a slice for this row/col
            slice = slices(sliceind);
        else
            % otherwise, set it to show black space below
            slice = slices(end) + 1;
        end

        if slice <= slices(end)
            % Get anatomy image
            anatIm = recomputeAnatImage(view,'anat',slice);

            % Get overlay
            overlay = [];
            if ~strcmp(mode,'anat')
                overlay = cropCurSlice(view,mode,slice);
            end

            % Select pixels that satisfy cothresh, phWindow, and mapWindow
            pts = [];
            if ~isempty(overlay)
                pts = ones(size(overlay));
                curCo = cropCurSlice(view,'co',slice);
                curPh = cropCurSlice(view,'ph',slice);
                curMap = cropCurSlice(view,'map',slice);

                if ~isempty(curCo) & cothresh>0
                    ptsCo = curCo > cothresh;
                    pts = pts & ptsCo;
                end

                if ~isempty(curPh)
                    if diff(phWindow) > 0
                        ptsPh = (curPh>=phWindow(1) & curPh<=phWindow(2));
                    else
                        ptsPh = (curPh>=phWindow(1) | curPh<=phWindow(2));
                    end
                    pts = pts & ptsPh;
                end

%               if strcmp(view.ui.displayMode, 'amp')
%                 curAmp = cropCurSlice(view, 'amp', slice);
%                 mnv = min(curAmp(:));
%                 mxv = max(curAmp(:));
%                 curMap = (curAmp - mnv) ./ (mxv - mnv);
%               end

                if ~isempty(curMap)
                    ptsMap = (curMap>=mapWindow(1) & curMap<=mapWindow(2));
                    pts = pts & ptsMap;
                end
            end

            % Rescale overlay to [numGrays:numGrays+numColors-1]
            if ~isempty(overlay)
                if strcmp(clipMode,'auto')
                    if ~isempty(find(pts));
                        clipMin = min(clipMin, min(overlay(pts)));
                        clipMax = max(clipMax, max(overlay(pts)));
                    else
                        clipMin = min(clipMin, min(overlay(:)));
                        clipMax = max(clipMax, max(overlay(:)));
                    end
                else
                    clipMin = min(clipMode);
                    clipMax = max(clipMode);
                end
                overlay=rescale2(overlay,[clipMin,clipMax],...
                    [numGrays+1,numGrays+numColors]);
            end

            % Combine overlay with anatomy image
            if ~isempty(overlay)
                % Combine them in the usual way
                im = anatIm;
                indices = find(pts);
                im(indices) = overlay(indices);
            else
                % No overlay.  Just show anatomy image.
                im = anatIm;
            end

            % zoom
            zoom = round(zoom);
            im = im(zoom(3):zoom(4), zoom(1):zoom(2));
        else
            % there may be blank spaces at the end of the montage image
            im = zeros(size(im));
        end

        rowIm = [rowIm im];
    end

    montage = [montage; rowIm];
end


if isfield(view,'ui') & ~isequal(view.name,'hidden')
    if isempty(overlay)
        view.ui.cbarRange = [];
    else
        view.ui.cbarRange = [clipMin clipMax];
    end

    % show the montage in the view window
    figure(view.ui.windowHandle);
    cmap = eval(['view.ui.' mode 'Mode.cmap']);
    image(montage);
    colormap(cmap);
    axis image;
    axis off;
    montage = ind2rgb(montage,cmap); % make truecolor
    imshow(montage);

    % set as the view's image
    view.ui.image = montage;
end

return
