function h = outline(pts, prefs);
% Outline pixels / points on an image.
%
% Usage: h = outline(pts, <prefs>);
%
% pts is a 2xN numeric vector where the first row is the X location
% of each point, and the second row is the Y location. If values in
% pts are non-integer, they're rounded off.
%
% prefs is an optional struct in which preference arguments are passed.
% Below are valid field names for prefs, what they control, and their
% default value if omitted.
%   prefs.axesHandle: axes on which to draw points [current axes]
%   prefs.color: color of outline (HELP PLOT for format) [blue]
%   prefs.lineWidth: width of lines in outline [0.5]
%   prefs.method: drawing method for outline, an integer:
%                 (1) draw outlines of the perimeter
%                 (2) draw outlines of each pixel
%                   [default is 1, perimeter outline]
%
% Returns h, a handle to the lines.
%
%
% ras, 08/23/2005.
if ~exist('pts','var') | isempty(pts), return; end

%%%%%default params
axesHandle = []; % will replace w/ gca if not assigned below (this avoids
                 % creating unnecessary axes on control figures)
color = 'b';
lineWidth = 0.5;
lineStyle = '-';
method = 1;
w = 0.5; % offset from center of pixel

%%%%%parse prefs
if exist('prefs','var') & isstruct(prefs)
    for i = fieldnames(prefs)'
        eval(sprintf('%s = prefs.%s; ',i{1},i{1}));
    end
end

% if no axes selected, use gca
if isempty(axesHandle), axesHandle = gca; end

%%%%%round pts and remove redundant pts
pts = round(pts);
pts = intersectCols(pts,pts); % removes redundant

%%%%%make axes active and get ready to draw
axes(axesHandle);
hold on

x=pts(1,:);
y=pts(2,:);
h = [];

%%%%%draw points
if method==1
    % DRAW PERIMETER ONLY
    for i=1:size(pts,2)
        xMinus = find(x == x(i)-1);
        xEquals = find(x == x(i));
        xPlus = find(x == x(i)+1);
        if isempty(xMinus)
            h(end+1) = line([y(i)-w, y(i)+w],[x(i)-w, x(i)-w]);
        else
            if ~any(y(i) == y(xMinus))
                h(end+1) = line([y(i)-w, y(i)+w],[x(i)-w, x(i)-w]);
            end
        end
        if isempty(xPlus)
            h(end+1) = line([y(i)-w,y(i)+w],[x(i)+w, x(i)+w]);
        else
            if ~any(y(i) == y(xPlus))
                h(end+1) = line([y(i)-w,y(i)+w],[x(i)+w, x(i)+w]);
            end
        end
        if ~isempty(xEquals)
            if ~any(y(i) == y(xEquals)-1)
                h(end+1) = line([y(i)+w,y(i)+w],[x(i)-w, x(i)+w]);
            end

            if ~any(find(y(i) == y(xEquals)+1))
                h(end+1) = line([y(i)-w,y(i)-w],[x(i)-w, x(i)+w]);
            end
        end
    end
else
    % DRAW BOXES AROUND EACH PIXEL
    for i=1:size(pts,2)
        h(end+1) = line([y(i)-w, y(i)+w, y(i)+w, y(i)-w, y(i)-w],...
                        [x(i)-w, x(i)-w, x(i)+w, x(i)+w, x(i)-w]);

    end
end

%%%%%set color, line width
set(h,'Color',color,'LineWidth',lineWidth,'LineStyle',lineStyle);

return
