function fullName = mrvSelectFile(rw,ext,dataType,windowTitle,startDir)
% Select a file name; the name can be to read or write
%
%  fullName = mrvSelectFile([rw],[ext],[dataType],[windowTitle],[startDir])
%
%Purpose: 
%   The user is prompted to select a file in a directory.  The directory
%   name is a persistent variable, so the next time this routine is called,
%   the user starts in the last selected directory.
%
%   use rw = 'r' for reading and rw = 'w' for writing.
%   Default is read.  
%   You may also pass in an extension to use for filtering file names.
%   Returns fulName = [] on Cancel.
%
%
% Examples
%  fullName = mrvSelectFile('r');
%  fullName = mrvSelectFile('r','mat',[],'Save Mesh',fileparts(getVAnatomyPath));
%

% Possibly, we should enforce the extension on the returned file name?

if ieNotDefined('rw'), rw = 'r'; end
if ieNotDefined('dataType'), dataType = []; end
if ieNotDefined('ext'), ext = '*'; end
if ieNotDefined('windowTitle'), windowTitle = ''; end

curDir = pwd;

% We remember the last directory the user chose. On the first call, this
% variable is empty.  When it is empty, the user can send in a start
% directory that sets the initial value.  The start directory is ignored
% later because pDir is no longer empty.
persistent pDir;
if isempty(pDir) && ~ieNotDefined('startDir'), pDir = startDir; end

if isempty(pDir)
    if ~isempty(dataType)
        % If there is no persistent directory, then try switching into the
        % directory associated with the specified data type.
        switch lower(dataType)
            case {'chooseFullPath according to dataType some day'}
                fullPath = pwd;  % for now, just pwd.
            otherwise
                fullPath = pwd;
        end
    end
    
    % If that directory doesn't exist, stay where you are
    if ieNotDefined('fullPath'), fullPath = pwd; end
else
    fullPath = pDir;
end

chdir(fullPath);
fileFilter = ['*.',ext];

switch lower(rw)
    case 'r'
        if(isempty(windowTitle)), windowTitle = 'MRV: Read Data'; end
        [fname, pname] = uigetfile(fileFilter, windowTitle);
    case 'w'
        if(isempty(windowTitle)), windowTitle = 'MRV: Write Data'; end
        [fname, pname] = uiputfile(fileFilter, windowTitle);
    otherwise
        error('Read/Write set incorrectly')
end

chdir(curDir)

% If the user pressed cancel, clean up the mess and go home.
if isequal(fname,0) | isequal(pname,0), fullName = [];
else fullName = fullfile(pname,fname); pDir = pname;
end


return;
