function [outStruct, ok] = generalDialog(uiStruct,title,figPos)
%
% [outStruct, ok] = generalDialog(uiStruct, [title], [figPos])
%
% Create a dialog w/ UI control text, edit, popup,
% checkbox, or listbox menus for different items,
% and return a struct with the user-selected values
% in the appropriate fields. A replacement for generaldlg.
%
% uiStruct: struct array w/ the following fields:
%   string: text to describe what item you want to set
%   fieldName: name of field in outStruct
%   list: For popup menus and listboxes, a cell array of strings
%         containing a list of possible options
%   style: What type of uicontrol should be: any of
%          'text', 'edit', 'popup', 'listbox', 'checkbox'
%   value: Default value for the control. For listbox and popups,
%          this can be a string specifying the name of the 
%          selected item, or a numeric index into the list;
%          for text and edit controls, this should be a string;
%          for checkbox, this can be a 1 or 0 (checked or not).
%
%  All other fields are ignored.
%  Each entry in the uiStruct reflects one item the
% user should select -- e.g., the name, age, and sex of a
% person (which might be described by two edit fields and a
% popup menu).
%
% outStruct is a struct with one field for each item. The
% name is determined by each item's 'fieldName' field in
% the uiStruct. Edit and text items return strings (text 
% items are not changed by the user); checkboxes return a 
% 1 or 0, depending on if the checkbox is selected or not;
% popups return a string containing the name or the 
% selected item; listbox returns a cell containing the 
% names of all selected items (multiple selection is allowed).
%
% Example:
% uiStruct(1).style = 'edit';
% uiStruct(1).string = 'Name of your chicken:';
% uiStruct(1).value = 'Ralph';
% uiStruct(1).fieldName = 'chickenName';
% 
% uiStruct(2).style = 'checkbox';
% uiStruct(2).string = 'neutered?';
% uiStruct(2).value = 1;
% uiStruct(2).fieldName = 'catNeuteredFlag';
% 
% uiStruct(3).style = 'popup';
% uiStruct(3).string = 'sex?';
% uiStruct(3).list = {'Male' 'Female' 'Neutered'};
% uiStruct(3).value = 'Male';
% uiStruct(3).fieldName = 'chickenSexFlag';
% 
% uiStruct(4).style = 'edit';
% uiStruct(4).string = 'IQ:';
% uiStruct(4).value = 100.34;
% uiStruct(4).fieldName = 'catIQ';
% 
% uiStruct(5).style = 'listbox';
% uiStruct(5).string = 'Favorite toy?';
% uiStruct(5).list = {'grocery bag' 'catnip' 'rubber ball'};
% uiStruct(5).value = 'catnip';
% uiStruct(5).fieldName = 'catToy';
% 
% outStruct = generalDialog(uiStruct)
%
% ras, started 04/08/05. In progress.
% ras, 05/05: working nicely.
% ras, 06/05: listbox now accepts multiple selection.
% ras, 06/30/05: imported into mrVista 2.0 Test repository.
if ieNotDefined('title'),    title = 'mrVista';         end

outStruct = [];
ok = 0;

%%%%%%%%%%%%%%
% Parameters %
%%%%%%%%%%%%%%
nItems = length(uiStruct);
font = 'Helvetica';
if isunix, fontsz = 10; else, fontsz = 9; end
figColor = [.9 .9 .9]; % background color for figure

% make all style fields lowercase, use ok keywords
for i = 1:nItems
    uiStruct(i).style = lower(uiStruct(i).style);
    switch uiStruct(i).style(1:3),
        case 'tex', uiStruct(i).style = 'text';
        case 'edi', uiStruct(i).style = 'edit';
        case 'pop', uiStruct(i).style = 'popup';
        case 'lis', uiStruct(i).style = 'listbox';
        case 'che', uiStruct(i).style = 'checkbox';
        otherwise, error('Uknown uiStruct style')
    end
end

% figure out # of lines for diff't items
nLines = 0;
for i = 1:nItems
    if strncmp(uiStruct(i).style,'list',4)
        nLines = nLines + length(uiStruct(i).list);
    else
        nLines = nLines + 1;
    end
end

% add 2 lines for 'Ok','Cancel' buttons plus spacers
nLines = nLines + 2;


if ieNotDefined('figPos')
	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	% Figure out how large the figure should            %
	% be, based on how many lines need to be displayed: %
	% (all controls except listboxes will use 1 line,   %
	% listboxes will use as many as there are options   %
	% in the list.)                                     %
	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
	% now figure out a good height, normalized to the screen
	figHeight = .04 * nLines;
	figHeight = min(figHeight,.8); % Don't get too big
	
	% if there's an existing figure, we'll set the dialog
	% to be centered in that window; otherwise, we'll
	% choose a reasonable start location
	if ~isempty(get(0,'CurrentFigure'))
        % fig exists, use position
        set(gcf,'Units','Normalized');
        ref = get(gcf,'Position');
        figWidth = 0.75 * ref(3);
        corner(1) = ref(1) + 0.125*ref(3); % roughly center in fig
        corner(2) = ref(2) + ref(3)/2 - figHeight/2;
	else
        % no fig, choose reasonable defaults
        figWidth = 0.6;
        corner = [0.2 0.2];
	end
        
	figPos = [corner figWidth figHeight];
end

% figure a size for the height of each
% item -- shouldn't be too high:
height = 1/nLines;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Open the Figure          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
hfig = figure('Name',title,...
    'Units','Normalized',...
    'Position',figPos,...
    'NumberTitle','off',...
    'Color',figColor);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Make controls for each item %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
bottom = 1-height; % initial bottom line for controls
for i = 1:nItems
    % (Note, all units are normalized to figure size)

    % Make a text label describing the item:
    % (If relevant
    if ismember(uiStruct(i).style, {'edit' 'listbox' 'popup' 'text'})
        textPos = [.1 (1-i*height) .4 height];
        uicontrol('Style','text','String',uiStruct(i).string,...
            'ForegroundColor','k','BackgroundColor',figColor,...
            'FontSize',fontsz,'FontName',font,'FontAngle','normal',...
            'FontWeight','bold','HorizontalAlignment','left',...
            'Units','Normalized','Position',textPos);
    end

    % determine item control, based on style
    if ismember(uiStruct(i).style, {'checkbox'})
        itemPos = [.5 bottom .4 height];
    elseif ismember(uiStruct(i).style,{'edit' 'popup' 'text' })
        itemPos = [.5 bottom .4 height];
    else
        % style should be 'listbox', many lines
        nChoices = length(uiStruct(i).list);
        itemPos = [.5 (1-(i+nChoices-1)*height) .4 (nChoices*0.9/nLines)];
    end
    bottom = bottom - itemPos(4); % move bottom of next control down

    % make a control for the item value
    switch uiStruct(i).style
        case 'text',
            h(i) = uicontrol('Style','text','String',uiStruct(i).value,...
                'ForegroundColor','k','BackgroundColor',figColor,...
                'FontSize',fontsz,'FontName',font,...
                'Units','Normalized','Position',itemPos);

        case 'edit',
            h(i) = uicontrol('Style','edit','String',uiStruct(i).value,...
                'ForegroundColor','k','BackgroundColor',figColor,...
                'FontSize',fontsz,'FontName',font,...
                'Units','Normalized','Position',itemPos);

        case 'checkbox',
            h(i) = uicontrol('Style','checkbox','String',uiStruct(i).string,...
                'ForegroundColor','k','BackgroundColor',figColor,...
                'FontSize',fontsz,'FontName',font,'FontAngle','italic',...
                'FontWeight','bold','Value',uiStruct(i).value,...
                'Units','Normalized','Position',itemPos);

        case 'listbox',
            if ischar(uiStruct(i).value)
                % find character default value in list of options
                val = cellfind(uiStruct(i).list,uiStruct(i).value);
            else
                % numeric default value--index into list
                val = uiStruct(i).value;
            end
            h(i) = uicontrol('Style','listbox','String',uiStruct(i).list,...
                'ForegroundColor','k','BackgroundColor',figColor,...
                'FontSize',fontsz,'FontName',font,...
                'Value',val,'Max',4,'Min',1,...
                'Units','Normalized','Position',itemPos);

        case 'popup',
            if ischar(uiStruct(i).value)
                % find character default value in list of options
                val = cellfind(uiStruct(i).list,uiStruct(i).value);
            else
                % numeric default value--index into list
                val = uiStruct(i).value;
            end
            if isempty(val), val = 1;   end
            h(i) = uicontrol('Style','popupmenu','String',uiStruct(i).list,...
                'ForegroundColor','k','BackgroundColor',figColor,...
                'FontSize',fontsz,'FontName',font,...
                'Value',val,...
                'Units','Normalized','Position',itemPos);

        otherwise,
            warning('generaldlg2: invalid style specified...')
            uiStruct(i)
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%
% Add OK/Cancel Buttons %
%%%%%%%%%%%%%%%%%%%%%%%%%
% OK button
uicontrol('Style','pushbutton','String','OK',...
    'ForegroundColor','k','BackgroundColor',[0 .7 0],...
    'ForegroundColor','w',...
    'FontSize',fontsz,'FontName',font,...
    'Callback','uiresume;','Tag','OK',...
    'Units','Normalized','Position',[.04 height/2 .2 height]);
% Cancel button
uicontrol('Style','pushbutton','String','Cancel',...
    'ForegroundColor','k','BackgroundColor',[.7 0 0],...
    'FontSize',fontsz,'FontName',font,...
    'Callback','uiresume;','Tag','Cancel',...
    'Units','Normalized','Position',[.76 height/2 .2 height]);

response = '';

% wait for user to select values
% wait for a 'uiresume' callback from OK/Cancel
uiwait;

%determine which button was hit.
response = get(gco,'Tag');

% parse the response
if ~isequal(response,'OK')
    % exit quietly
    close(hfig);
    return
else
    ok = 1; % we got a response, will parse below
end

% if we got here, we should be able to proceed
for i = 1:length(h)
    field = uiStruct(i).fieldName;
    switch uiStruct(i).style
        case 'text', outStruct.(field) = get(h(i),'String');
        case 'edit', outStruct.(field) = get(h(i),'String');
        case 'checkbox', outStruct.(field) = get(h(i),'Value');
        case 'listbox', 
            outStruct.(field) = uiStruct(i).list(get(h(i),'Value'));
        case 'popup', 
            outStruct.(field) = uiStruct(i).list{get(h(i),'Value')};
    end
end

% close the figure
close(hfig);

return





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%debug
uiStruct(1).style = 'edit';
uiStruct(1).string = 'Name of your chicken:';
uiStruct(1).value = 'Ralph';
uiStruct(1).fieldName = 'chickenName';

uiStruct(2).style = 'checkbox';
uiStruct(2).string = 'neutered?';
uiStruct(2).value = 1;
uiStruct(2).fieldName = 'catNeuteredFlag';

uiStruct(3).style = 'popup';
uiStruct(3).string = 'sex?';
uiStruct(3).list = {'Male' 'Female' 'Neutered'};
uiStruct(3).value = 'Male';
uiStruct(3).fieldName = 'chickenSexFlag';

uiStruct(4).style = 'edit';
uiStruct(4).string = 'Name of your cat:';
uiStruct(4).value = 'Fluffy';
uiStruct(4).fieldName = 'catName';

uiStruct(5).style = 'checkbox';
uiStruct(5).string = 'neutered?';
uiStruct(5).value = 1;
uiStruct(5).fieldName = 'chickenNeuteredFlag';

uiStruct(6).style = 'popup';
uiStruct(6).string = 'sex?';
uiStruct(6).list = {'Male' 'Female' 'Neutered'};
uiStruct(6).value = 'Female';
uiStruct(6).fieldName = 'catSexFlag';

uiStruct(7).style = 'edit';
uiStruct(7).string = 'IQ:';
uiStruct(7).value = 100.34;
uiStruct(7).fieldName = 'catIQ';

uiStruct(8).style = 'listbox';
uiStruct(8).string = 'Favorite toy?';
uiStruct(8).list = {'grocery bag' 'catnip' 'rubber ball'};
uiStruct(8).value = 'catnip';
uiStruct(8).fieldName = 'catToy';

outStruct = generalDialog(uiStruct)

