function view = roiSetOptions(view,opts);
%
% view = roiSetOptions(view,[opts]);
%
% Set options/prefs related to rendering
% of ROIs in mrVista. Right now this includes
% the color of the selcted ROI, the format
% for rendering (boxes surrounding pixels,
% perimeter, thick perimeter), and the
% number of ROIs to view. Down the line, 
% this may be a better way to keep track 
% of how ROIs are viewed.
%
% If called with <2 input args, will pop
% up a dialog to get these prefs. But, you
% can pass in an opts struct, in which case
% no dialog will be put up and the options
% will be set from fields in opts. The following
% fields are looked for: 
%       selRoiColor: color vector [R G B], from 0-1 for each,
%                    or string (e.g.,'r','w','k') -- see help plot
%       showRois:    integer from 0-2:
%                    0: hide all ROIs
%                    1: show selected ROI
%                    2: show all ROIs
%       drawFormat:  integer from 1-3:
%                    1: use boxes around each pixel
%                    2: draw perimeter
%                    3: draw filled perimeter (looks nicer on some views 
%                                              like FLAT)
%          
%
% The settings are placed in the following
% fields in the view's ui substruct:
%   selected ROI color: selRoiColor
%   which ROIs to show: showROIs
%   rendering format: this is stored in both the 
%                     showRois and filledPerimeter
%                     fields. If either perimeter or
%                     filled perimeter is set, showROIs
%                     is negative (-1 show cur ROI perim,
%                     -2 show all ROIs perim). 
%                     filledPerimeter is 0 or 1, depending
%                     on whether to use the filled perimeter 
%                     option.
%
% Suggestions for down the line:
%       1) We should consolidate ROI-related UI info in a view into
%          a substruct like ui.roi. This should contain the info 
%          set here, as well as the pointers to the ROI popup, 
%          and additional info like the name of the current ROI
%          (w/ space to add more info)
%       2) I prefer separating the settings for which ROIs to show
%          and the rendering format as I use for the opts struct:
%          there should be space to add new rendering formats easily,
%          since even the existing 3 options don't always look great.
%          Might want to set things like line width, etc.
%
%
%
% ras 05/05.
if ieNotDefined('view')
    view = getSelectedInplane;
end

% check for hidden view
if isequal(view.name,'hidden') | ~checkfields(view,'ui')
    error('This appears to be a hidden view.')
end

% get a selected ROI color: if it's not
% present in the view, set as default
ui = viewGet(view,'ui');
if checkfields(ui,'selRoiColor')
    selRoiColor = ui.selRoiColor;
else
    selRoiColor = [0 0 1]; % blue default
end

% get the currently-selected option for showing
% ROIs
showRois = ui.showROIs;

% get draw format -- infer from showROIs and 
% filledPerimeter fields
if checkfields(ui,'filledPerimeter')
    fill = ui.filledPerimeter;
else
    fill = 0;
end
if showRois>=0
    drawFormat = 1;  % boxes
else
    drawFormat = 2 + fill; % perim / filled perim
end

if ieNotDefined('opts')
    % user dialog
    inpt(1).fieldName = 'selRoiColor';
    inpt(1).style = 'edit'; 
    inpt(1).string = 'Selected ROI Color:';
    inpt(1).value = num2str(selRoiColor);

    inpt(2).fieldName = 'showRois';
    inpt(2).style = 'popup'; 
    inpt(2).string = 'Show which ROIs?';
    inpt(2).list = {'None (Hide ROIs)' 'Selected' 'All'};
    inpt(2).value = abs(showRois)+1;

    inpt(3).fieldName = 'drawFormat';
    inpt(3).style = 'popup'; 
    inpt(3).string = 'ROI Rendering Method:';
    inpt(3).list = {'Boxes Around Each Pixel' ...
                  'Perimeter' ...
                  'Filled Perimeter'};
    inpt(3).value = drawFormat;

    opts = generalDialog(inpt,'ROI Set Options');

    % if user quit, exit gracefully
    if isempty(opts)
        return
    end

    % parse responses
    if ~isempty(str2num(opts.selRoiColor))
        opts.selRoiColor = str2num(opts.selRoiColor);
    end
    opts.showRois = cellfind(inpt(2).list,opts.showRois)-1;
    opts.drawFormat = cellfind(inpt(3).list,opts.drawFormat);    
end

% set fields appropriately
if isfield(opts,'selRoiColor')
    ui.selRoiColor = opts.selRoiColor;
end

ui.showROIs = opts.showRois;
if opts.drawFormat>1
    % perim or filled perim
    ui.showROIs = -1 * ui.showROIs;
end

ui.filledPerimeter = (opts.drawFormat==3);

view = viewSet(view,'ui',ui);

% let's go ahead and save the viewing preferences
% as well
savePrefs(view);

disp('Updated ROI Viewing Options.')

refreshScreen(view);

return



    
