function amps = mv_amps(mv,runs);
%
% amps = mv_amps(mv,[runs]);
%
% Return a 2D matrix of format
% voxels x conditions containing
% response amplitudes, evaluated 
% according to the 'ampType' parameter
% set in the params struct.
%
% ras, 05/05.
if ieNotDefined('mv')
    mv = get(gcf,'UserData');
end

allRuns = unique(mv.trials.run);

if ieNotDefined('runs')
    runs = allRuns;
end

% check that we have an amplitude type defined
if ~checkfields(mv,'params','ampType')
    % set up a dialog, get it
    ui.string = 'Method to Calculate Amplitudes?';
    ui.fieldName = 'ampType';
    ui.list = {'Peak-Bsl Difference', 'GLM Betas', ...
        'Dot-product Relative Amps'};
    ui.style = 'popup';
    ui.value = ui.list{1};
    
    resp = generaldlg2(ui,'Select Amplitude Type');
    ampInd = cellfind(ui.list,resp.ampType);
    opts = {'difference' 'betas' 'relamps'};
    mv.params.ampType = opts{ampInd};
end

switch mv.params.ampType
    case 'difference',   
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % peak-bsl difference              %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        amps = squeeze(nanmeanDims(mv.voxAmps(runs,:,:),1));
        
    case 'betas',       
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%        
        % apply a GLM and get beta values  %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % Part 1: Need to find rows in tSeries 
        % corresponding to the selected runs:
        %
        tr = mv.params.framePeriod;
        nConds = sum(mv.trials.condNums>0);
        nFrames = size(mv.tSeries,1);

        % first construct an index, for each frame of the tSeries,
        % of the run that frame came from:        
        runIndex = zeros(1,nFrames);
        
        firstTrials = find(nVals(mv.trials.run)==1); 
        firstFrames = mv.trials.onsetFrames(firstTrials);
        runIndex(firstFrames) = mv.trials.run(firstTrials);    

        % fill in each unassigned entry in the index
        % with the current run (given in the previous entry):
        I = intersect(firstFrames+1,1:nFrames); % restrict to max # frames
        while any(runIndex(I)==0)
            runIndex(I) = runIndex(I-1);
            I = intersect(I+1, 1:nFrames);
        end
        
        % now grab the appropriate rows from the tSeries:
        ind = find(ismember(runIndex, runs));
        tSeries = mv.tSeries(ind,:);

        % part 2: get event onset/condition data
        % for selected runs in subtrials struct: 
        %
        subtrials = mv.trials;
        if ~isequal(runs,allRuns)
            % subtrials needs to reflect only selected runs, 
            % (tricky):
            
            % First, find the # of frames in each run:
            lastTrials = [firstTrials(2:end)-1 length(mv.trials.onsetFrames)];
            lastTrialFrames = mv.trials.onsetFrames(lastTrials);
            framesPerRun = [lastTrialFrames(1) diff(lastTrialFrames)];
            
            % Next, select event data only for selected subsets:
            ok = find(ismember(subtrials.run,runs));
            subtrials.onsetSecs = subtrials.onsetSecs(ok);
            subtrials.onsetFrames = subtrials.onsetFrames(ok);
            subtrials.cond = subtrials.cond(ok);
            subtrials.label = subtrials.label(ok);
            subtrials.run = subtrials.run(ok);
            subtrials.parfiles = subtrials.parfiles(runs);
            
            % Lastly, correct the onset information: since onsets
            % are counted cumulatively, they will reflect
            % non-selected runs. Correct for this overcounting:
            skippedRuns = setdiff(allRuns,runs);
            for j = skippedRuns
                offset = framesPerRun(j);
                laterRuns = find(subtrials.run>j);
                subtrials.onsetFrames(laterRuns) = ...
                    subtrials.onsetFrames(laterRuns) - offset;
                subtrials.onsetSecs(laterRuns) = ...
                    subtrials.onsetSecs(laterRuns) - offset*tr;
            end
        end
                
        
        % Part3: Build a design matrix, apply the glm, grab betas
        %
        [X, nh, hrf] = glm_createDesMtx(subtrials,mv.params,tSeries,0);
                
        model = glm(double(tSeries), X, tr, nh);       
        
        amps = permute(model.betas(1,1:nConds,:), [3 2 1]);
        
    case 'relamps',
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%        
        % dot-product relative amplitdues: %
        % * not currently implemented *    %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%     
        amps = er_relamps(mv.voxData(:,runs,:,:));
        amps = squeeze(nanmean(amps));
end

% return amplitudes for selected conditions only
sel = find(tc_selectedConds(mv));
amps = amps(:,sel-1);


return