function params = er_getParams(view,scan,dt);
% params = er_getParams(view,[scan],[dt]):
%
% Get the event-related analysis params
% for the current scan, stored in dataTYPES'
% eventAnalysisParams subfield. If some of 
% these fields are absent, initialize them 
% to reasonable guess values. Returns params
% in a 'params' struct. 
%
% A lot of the newer params are used for
% er_chopTSeries, and the time course UI.
%
% scan defaults to view's current scan.
%
% ras 12/04
global dataTYPES;

if ieNotDefined('view')
    view = getSelectedInplane;
end

if ieNotDefined('scan')
    scan = getCurScan(view);
end

if ieNotDefined('dt')
    dt = viewGet(view,'curDataType');
end

if ischar(dt), dt = existDataType(dt); end
dtName = dataTYPES(dt).name;

params = dataTYPES(dt).eventAnalysisParams(scan);

defaults = er_defaultParams;

% flag to zero baseline or not
if ~isfield(params,'normBsl') | isempty(params.normBsl)
	params.normBsl = defaults.normBsl;         
end

% threshold for significant activations
if ~isfield(params,'alpha') | isempty(params.alpha)
	params.alpha = defaults.alpha;           
end

% period to use as baseline in t-tests, in seconds
if ~isfield(params,'bslPeriod') | isempty(params.bslPeriod)
	params.bslPeriod = defaults.bslPeriod;       
end

% period to look for peaks in t-tests, in seconds
if ~isfield(params,'peakPeriod') | isempty(params.peakPeriod)
	params.peakPeriod = defaults.peakPeriod;       
end

% seconds relative to trial onset to take for each trial
if ~isfield(params,'timeWindow') | isempty(params.timeWindow)
	params.timeWindow = defaults.timeWindow;     
end

% conditions to use for calculating signal-to-noise, HRF
if ~isfield(params,'snrConds') | isempty(params.snrConds)
	params.snrConds = defaults.snrConds;     
end


% # secs to shift onsets in parfiles, relative to time course
if ~isfield(params,'onsetDelta') | isempty(params.onsetDelta)
	params.onsetDelta = defaults.onsetDelta;         
end

% flag for which hemodynamic impulse response 
% function to use if applying a GLM:
% -------------------------------------------
% 0: deconovolve (selective averaging)
% 1: estimate HRF from mean response to all non-null conditions
% 2: Use Boynton et all 1998 gamma function
% 3: Use SPM difference-of-gammas
% 4: Use HRF from Dale and Buckner, 1997 (very similar to Boynton
%    gamma)
% OR, if flag is a char: name of a saved HRF function
%    (stored in subject/HRFs/, where subject is the subject's
%     anatomy directory, where the vAnatomy is stored)
if ~isfield(params,'glmHRF') | isempty(params.glmHRF)
	params.glmHRF = defaults.glmHRF;         
end

% flag for whether or not to estimate temporally-correlated
% noise in data when applying a GLM, referred to as 'whitening':
% (see Dale and Burock, HBM, 2000):
if ~isfield(params,'glmWhiten') | isempty(params.glmWhiten)
    glmWhiten = 0;
end

% detrend flag: 
%--------------
% -1 linear detrend, 0 no detrend, 1 multiple boxcar smoothing,
% 2 quartic trend removal
if ~isfield(params,'detrend') | isempty(params.detrend)
	params.detrend = defaults.detrend;         
end

% detrend frames: 
% for detrend option 1, # of frames for smoothing
% kernal (roughly equivalent to cutoff of highpass
% filter):
if ~isfield(params,'detrendFrames') | isempty(params.detrendFrames)
	params.detrendFrames = defaults.detrendFrames;         
end

% inhomogeneity correction flag:     
%-------------------------------
%     0 do nothing
%     1 divide by the mean, independently at each voxel
%     2 divide by null condition
%     3 divide by anything you like, e.g., robust estimate of intensity
%     inhomogeneity
if ~isfield(params,'inhomoCorrect') | isempty(params.inhomoCorrect)
	params.inhomoCorrect = defaults.inhomoCorrect;         
end


% temporal normalization flag: 
if ~isfield(params,'temporalNormalization') | isempty(params.temporalNormalization)
	params.temporalNormalization = defaults.temporalNormalization;         
end

% frame period:
% this is redundant w/ the scan params, but since
% I use these event params independent of mrVista and
% dataTYPES, we'll want it here too:
if ~isfield(params,'framePeriod')  | isempty(params.framePeriod)
    params.framePeriod = dataTYPES(dt).scanParams(scan).framePeriod;
end

% parfiles:
% this is also redundant w/ the scan params, but is essential
% for event-related analyses
if ~isfield(params,'parfiles')  | isempty(params.parfiles)
    params.parfiles = dataTYPES(dt).scanParams(scan).parfile;
end

return