function params = er_editParams(params, dtName, scans);
% A dialog to edit event-related analysis parameters.
%
% params = er_editParams(params, [dtName, scans]);
%
% This routine DOES NOT save or set them; use er_setParams
% to do that.
%
% If the data type name and scan numbers are passed as arguments,
% will amend the dialog title text to clarify which parameters are
% being edited
% 
%
% ras, 09/2005.
if notDefined('params'),  params = er_defaultParams; end

%%%%%check that all fields are assigned
def = er_defaultParams;
unassigned = setdiff(fieldnames(def), fieldnames(params));
for f = unassigned(:)'
    params.(f{:}) = def.(f{:});
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% set up a dialog with generaldlg2:          %
% each parameter to set will be described    %
% by its control settings                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
dlg = struct('fieldName','','style','','list',{},'value','',...
             'string','');

% seconds relative to trial onset to take for each trial
dlg(end+1).fieldName = 'timeWindow';
dlg(end).style = 'edit';
dlg(end).list = {};
dlg(end).value = num2str(params.(dlg(end).fieldName));
dlg(end).string = 'Time Window, in seconds (incl. pre-stimulus onset period):';
       
% period to use as baseline for each event, in seconds
dlg(end+1).fieldName = 'bslPeriod';
dlg(end).style = 'edit';
dlg(end).list = {};
dlg(end).value = num2str(params.(dlg(end).fieldName));
dlg(end).string = 'Baseline Period, in seconds:';

% period to look for peak activation for each event, in seconds
dlg(end+1).fieldName = 'peakPeriod';
dlg(end).style = 'edit';
dlg(end).list = {};
dlg(end).value = num2str(params.(dlg(end).fieldName));
dlg(end).string = 'Peak Period, in seconds:';

% flag to zero baseline or not
dlg(end+1).fieldName = 'normBsl';
dlg(end).style = 'checkbox';
dlg(end).list = {};
dlg(end).value = params.(dlg(end).fieldName);
dlg(end).string ='Normalize all trials during the baseline period';

% threshold for T-test determination of significant activations
dlg(end+1).fieldName = 'alpha';
dlg(end).style = 'edit';
dlg(end).list = {};
dlg(end).value = num2str(params.(dlg(end).fieldName));
dlg(end).string = 'Alpha for significant activation (peak vs baseline):';

% # secs to shift onsets in parfiles, relative to time course
dlg(end+1).fieldName = 'onsetDelta';
dlg(end).style = 'edit';
dlg(end).list = {};
dlg(end).value = num2str(params.(dlg(end).fieldName));
dlg(end).string = 'Shift onsets relative to time course, in seconds:';

% conditions to use for calculating signal-to-noise, HRF
dlg(end+1).fieldName = 'snrConds';
dlg(end).style = 'edit';
dlg(end).list = {};
dlg(end).value = num2str(params.(dlg(end).fieldName));
dlg(end).string = 'Use which conditions for computing SNR / HRF?';

% flag for which hemodynamic impulse response 
% function to use if applying a GLM:
% -------------------------------------------
% 0: deconovolve (selective averaging)
% 1: estimate HRF from mean response to all non-null conditions
% 2: Use Boynton et all 1998 gamma function
% 3: Use SPM difference-of-gammas
% 4: Use HRF from Dale and Buckner, 1997 (very similar to Boynton
%    gamma)
% OR, if flag is a char: name of a saved HRF function
%    (stored in subject/HRFs/, where subject is the subject's
%     anatomy directory, where the vAnatomy is stored)dlg(end+1).fieldName = 'snrConds';
glmList = {'Deconvolve' 'Boynton Gamma' 'SPM Difference-of-gammas' ...
             'Dale & Buckner 99 HIRF' 'Choose from file...'};
dlg(end+1).fieldName = 'glmHRF';
dlg(end).style = 'popup';
dlg(end).list = glmList;
dlg(end).value = params.glmHRF+1;
dlg(end).string = 'HRF to use for GLM?';

% flag for whether or not to estimate temporally-correlated
% noise in data when applying a GLM, referred to as 'whitening':
% (see Dale and Burock, HBM, 2000):
dlg(end+1).fieldName = 'glmWhiten';
dlg(end).style = 'checkbox';
dlg(end).list = {};
dlg(end).value = params.(dlg(end).fieldName);
dlg(end).string = 'Estimate temporally-correlated noise (whiten) in GLMS';

% temporal normalization flag: 
dlg(end+1).fieldName = 'temporalNormalization';
dlg(end).style = 'checkbox';
dlg(end).list = {};
dlg(end).value = params.(dlg(end).fieldName);
dlg(end).string = 'Normalize each temporal volume in computing tSeries';

% Options for how to compensate for distance from the coil, depending
% on the value of inhomoCorrection 
%   0 do nothing
%   1 divide by the mean, independently at each voxel
%   2 divide by null condition
%   3 divide by anything you like, e.g., robust estimate of intensity inhomogeneity
% For inhomoCorrection=3, you must compute the spatial gradient
% (from the Analysis menu) or load a previously computed spatial 
% gradient (from the File/Parameter Map menu).
icList = {'Do nothing' 'Divide each voxel by the mean' ...
           'Divide by the null condition' ...
           'Divide by spatial gradient map'};
dlg(end+1).fieldName = 'inhomoCorrect';
dlg(end).style = 'popup';
dlg(end).list = icList;
dlg(end).value = params.(dlg(end).fieldName)+1;
dlg(end).string = 'Inhomogeneity Correction';


% flag to edit condition colors in another dialog
dlg(end+1).fieldName = 'assignCondColors';
dlg(end).style = 'checkbox';
dlg(end).list = {};
dlg(end).value = 0;
dlg(end).string = 'Assign Condition Colors (in another dialog)';

% flag to assign parfiles in another dialog
dlg(end+1).fieldName = 'assignParfiles';
dlg(end).style = 'checkbox';
dlg(end).list = {};
dlg(end).value = 0;
dlg(end).string = 'Assign .par files (in another dialog)';

%%%%%title for dialog
ttl = 'Edit Event-Related Parameters';
if exist('dtName', 'var') & exist('scans', 'var')
    ttl = [ttl sprintf(', %s Scans %s', dtName, num2str(scans))];
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% put up the dialog, get response   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
params = generalDialog(dlg, ttl, [.3 .2 .5 .7]);
if isempty(params), return; end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% parse response                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
params.timeWindow = str2num(params.timeWindow);
params.bslPeriod = str2num(params.bslPeriod);
params.peakPeriod = str2num(params.peakPeriod);
params.alpha = str2num(params.alpha);
params.onsetDelta = str2num(params.onsetDelta);
params.snrConds = str2num(params.snrConds);
params.glmHRF = cellfind(glmList,params.glmHRF)-1;
params.inhomoCorrect = cellfind(icList, params.inhomoCorrect) - 1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% if user requested, assign parfiles, condition colors %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% TODO: make this work!
if params.assignParfiles==1
    hI = initHiddenInplane;
    hI = er_assignParfilesToScans(hI); 
end

if params.assignCondColors==1
    trials = er_assignColors(er_concatParfiles);
    params.condColors = trials.condColors;
end
params = rmfield(params,'assignCondColors');



return
