function tc = tc_removeOutliers(tc, criterion, thresh, action);
%
% tc = tc_removeOutliers([tc], [criterion, thresh, action]);
%
% Remove outlier points / trials / events from a time course.
% 
% INPUTS:
% tc: time course struct.
% 
% criterion: flag for which criterion to use to determine outlier points.
%   0 -- find time points a certain number of standard deviations 
%       (abs. value) from the mean; 
%   1 -- find time points whose absolute value exceeds a certain threshold
%        value.
%
% thresh: value to use as the threshold (in # of standard deviations,
% if the criterion is 0, or the units of tc.wholeTc if criterion is 1).
%
% action: flag describing which action to take for outlier points:
%   0 -- interpolate between the surrounding time points.
%   1 -- remove the event during which the outlier data occurs from 
%        the trials struct. Does not modify the time course data itself,
%        but modifies the trials information.
%   2 -- append points to a field params.outliers subfield, containing
%        information on the outlier time and the event to which it
%        belonged, to be considered by future analyses. This doesn't
%        modify either the time course data or the trials information,
%        but I haven't yet written any code that parses this information.
%
% If any of the last 3 arguments are omitted, a dialog is brought up.
%
% 11/2/2005 by ras.
if ieNotDefined('tc'), tc = get(gcf,'UserData'); end

if ieNotDefined('criterion') | ieNotDefined('thresh') | ieNotDefined('action')
    % dialog
	ui(1).string = 'Threshold value:';
	ui(1).fieldName = 'thresh';
	ui(1).style = 'edit';
	ui(1).value = '3';

	ui(2).string = 'Criterion:';
	ui(2).fieldName = 'criterion';
	ui(2).list = {'Standard Deviations from Mean' 'Absolute % Signal'};
	ui(2).style = 'popup';
	ui(2).value = 1;
    
    ui(3).string = 'Action to take for outliers?';
    ui(3).fieldName = 'action';
    ui(3).style = 'popup';
	ui(3).list = {'Replace data point with interpolated neighbors' ...
                  'Remove event from trials struct' ...
                  'Mark outliers in params.outliers subfield'};
    ui(3).value = 1;
    
	resp = generalDialog(ui,'Remove Outliers');
    thresh = str2num(resp.thresh);
    criterion = cellfind(ui(2).list, resp.criterion)-1;
    action = cellfind(ui(3).list, resp.action);
end    

% find outliers, based on specified criterion
switch criterion
    case 0, % find time points w/ mean > thresh std. deviations from
        sigma = nanstd(abs(tc.wholeTc));
        mu = nanmean(abs(tc.wholeTc));
        outliers = find(tc.wholeTc > mu + thresh*sigma);
        
    case 1, % find time points greater than threshold value
        outliers = find(abs(tc.wholeTc) > thresh);
        
end

% if no outliers, hooray!
if isempty(outliers)
    disp('tc_removeOutliers: No Outliers! Hooray!')
    return
end

% check that not ALL time points are outliers...
if length(outliers)==length(tc.wholeTc)
    msg = ['Warning: Given the selected criteria, ALL time points '...
           'were found to be outliers! No action taken.'];
    mrMessage(msg);
    return
end

% do what the selected action says
switch action
    case 1, % remove point, replacing w/ average of neighbors
        % NAMING NOTE: I'd intended to write a separate function
        % 'tSeriesRemoveOutliers' which takes the outlier point and
        % interpolates between them -- also adding the ability to 
        % find outliers using the same thresh / criterion variables as
        % here -- but more general to work on tSeries of many voxels.
        % However, Guillaume has beat me to it, and already written
        % a function with this name for motionComp tools. I think 
        % his function -- which takes a base image -- could be combined
        % with my intended functionality, but only down the line.
        for i = outliers(:)'
            % find nearest non-outlier neighbors
            nFrames = length(tc.wholeTc);
            a = i-1; while ismember(a,outliers) & a>1, a=a-1; end
            b = i+1; while ismember(b,outliers) & b<nFrames, b=b+1; end
            tc.wholeTc(i) = 0.5 * (tc.wholeTc(a)+tc.wholeTc(b));
        end
        
        % re-chop the tSeries
        tc = tc_recomputeTc(tc,1);

        
    case 2, % remove entry for event in trials
        rmEvents = [];
        for i = outliers(:)'
            j = find(tc.trials.onsetFrames<i);
            if ~isempty(j), rmEvents = [rmEvents j(end)]; end
        end
        nEvents = length(tc.trials.cond);
        ok = setdiff(1:nEvents, unique(rmEvents));
        
        tc.trials.cond = tc.trials.cond(ok);
        tc.trials.onsetSecs = tc.trials.onsetSecs(ok);
        tc.trials.onsetFrames = tc.trials.onsetFrames(ok);
        tc.trials.label = tc.trials.label(ok);
        tc.trials.run = tc.trials.run(ok);
        
        % re-chop the tSeries
        tc = tc_recomputeTc(tc,1);
                        
    case 3, % note where the outliers are, but take no other action
        rmEvents = [];
        for i = outliers(:)'
            j = find(tc.trials.onsetFrames<i);
            if ~isempty(j), rmEvents = [rmEvents j(end)]; end
        end
    
        tc.params.outliers.frames = outliers;
        tc.params.outliers.secs = outliers .* tc.trials.TR;
        tc.params.outliers.events = rmEvents;
end

% report on results
fprintf('%i Outlier Time Points Found. \n',length(outliers));

% refresh
timeCourseUI;
    

return
