function model = glm(Y, X, tr, nh, whiten);
%
% model = glm(Y, X, [tr], [nh], [whiten]):
%
% Apply a general linear model to the time series in 
% Y, using the experimental design information contained
% in X. 
%
% Y: matrix of time series data, of size nFrames by nVoxels,
% where nFrames = number of time points and nVoxels = # voxels.
% Will apply GLM to all these time series in one step.
%
% X: The Design Matrix.
% The Design matrix can be obtained by computed from either
% glm_createDesMtx, or glm_deconvolution_matrix. 
% In the former case, you are applying an estimated hemodynamic 
% response function to form predictors for each condition,
% and getting back one beta for each condition. 
% In the latter, you are returned an estimated response shape for
% each condition, by applying multiple (nh) predictors
% for each conditions (the predictors get the response 
% at different times relative to trial onset). 
% 
% In any case, the format of X is a 2-D matrix with
% rows = MR frames by columns = predictors.
% 
% The GLM will return one beta value and error estimate for
% each predictor, grouping every nh predictors together (see
% below).
%
% Optional Arguments:
%
% nh: # of time points in the estimated response to each
% condition. If you are applying an HRF, this should be
% set to 1. [default 1] 
%
% tr: frame period, in secs. [default 1 sec]
%
% whiten: if 1, will 'whiten' data by estimating noise
% covariance across voxels; otherwise will not. [Default 0]
%
%
% The output model struct includes the following fields:
%   betas: estimated hemodynamic responses, 
%           size [nh by nPredictors by nVoxels]
%
%   residual: estimated error for each point in h_bar,
%           size [nFrames by nVoxels] 
%
%   stdevs: estimated standard deviation for each beta value
%           size [nh by nPredictors by nVoxels]
%
%   sems: standard error of the mean for beta values
%           size [nh by nPredictors by nVoxels ]
%
%   C: estimated noise covariance matrix. If whiten
%      is 0, this is a big identity matrix; if it's 1,
%      [describe]
%
%   voxDepNoise: voxel-dependent noise estimate
%           size [nVoxels by 1]
%
%   voxIndepNoise: voxel-independent component of autocorrelation
%       matrix of betas. voxDepNoise * voxIndepNoise = the
%       autocorrelation matrix.
%           size [nPredictors by nPredictors]
%           
%   dof: degrees of freedom of the fitting.
%
%
% For a reference on the noise covariance methods,
% see Burock and Dale, Human Brain Mapping, 11:249-260.
% (2000) or the Greve theory paper included w/ Freesurfer
% FS-FAST. Thanks to you guys for the background.
%
% Code written from scratch by ras and gb, early 2005.

% TODO:
%  Write modelSet, modelGet, modelCreate();
%

if nargin < 2,                  help glm, return;   end
if ~exist('tr') | isempty(tr),  tr = 1;             end
if ~exist('nh') | isempty(nh),  nh = 22;            end
if ~exist('whiten') | isempty(whiten),  whiten = 0; end

if ~isa(Y,'double'),            Y = double(Y);      end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% initialize model struct                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
model.betas = [];
model.residual = [];
model.stdevs = [];
model.sems = [];
model.nh = nh;
model.tr = tr;
model.C = [];
model.voxDepNoise = [];
model.voxIndepNoise = [];
model.designMatrix = [];
model.dof = [];
model.whiten = whiten;

if nh>1
    model.type = 'selective averaging';
else
    model.type = 'applied HRF';
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initialize: size check, count trials, etc       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
nFrames = size(Y,1);
nVoxels = size(Y,2);
nPredictors = size(X,2);

if size(X,1)~=nFrames
    fprintf('Rows in Y: %i, Rows in X: %i\n',size(Y,1),size(X,1));
    error('X and Y should have the same number of rows.')
end
    
model.trial_count = glm_trial_count(X,nh);

%
model.designMatrix = X;

% Compute Degrees of Freedom of model
% Rtmp = eye(nFrames) - X*pinv(X); % Residual Error Forming Matrix 
model.dof = size(Y,1) - rank(X); %trace(Rtmp);



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Estimate h_bar, assume white noise              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
model.betas = X\Y;

% Compute residual error of initial fitting
model.residual = Y - X*model.betas;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% If selected, determine noise distribution and re-apply model %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if whiten==1
	% Estimate noise covariance matrix 
	model.C = glm_determine_noise(Y,model.residual,tr);
	
	% Using noise covariance matrix, recompute estimated responses 
    R = chol(model.C);
    R = (R')^(-1);
    model.betas = (X'*R*X)\(X'*R*Y);
	
	% Recompute residual error  
	model.residual = Y - X*model.betas;
	
else
    % estimated noise covariance is an identity matrix
    model.C = eye(nFrames);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute factors of covariance matrix for the betas:          %
% sigma-squared is the voxel-dependent noise for each voxel,   %
% while voxIndepNoise is the voxel-independent covariance      %
% (Both on the rhs in Eq (16) of the Greve paper)              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Variance of residual error (voxel-dependent scalar)
% (Disabled, too memory-hungry, not hugely useful)
% tmp = diag(model.residual' * inv(model.C) * model.residual)';
% model.voxDepNoise = tmp ./ model.dof;

% Hemodynamic covariance matrix (voxel-independent matrix)
model.voxIndepNoise = inv(X'* model.C * X);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Estimate Std. Deviation, SEMs for each beta value %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% estimated residual variance, std deviation
eres_var = sum(model.residual.^2) / model.dof; 
eres_std = sqrt(eres_var); 
model.stdevs = sqrt((diag(model.voxIndepNoise).*diag(X'*X)) * eres_std.^2);

% get SEMs: 
tmp = repmat(model.trial_count(:),[1 nVoxels]);
model.sems = model.stdevs ./ sqrt(tmp);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Reshape if necessary: Group every nh predictors %
% together in the main matrices                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if nh > 1
    % there may be baseline predictors appended to
    % the design matrix: group these separately
    % NOTE: this assumes the number of DC predictors
    % is less than nh; if this is no longer a realistic
    % assumption, will need to fix this:
    nDC = mod(nPredictors,nh);
    if nDC > 0        
        dcRange = nPredictors-nDC+1:nPredictors;

        model.dc_betas = model.betas(dcRange,:);
        model.dc_stdevs = model.stdevs(dcRange,:);
        model.dc_sems = model.sems(dcRange,:);
        
        model.betas = model.betas(1:dcRange(1)-1,:);
        model.stdevs = model.stdevs(1:dcRange(1)-1,:);
        model.sems = model.sems(1:dcRange(1)-1,:);
    end
    
end

nConds = floor(nPredictors/nh);
model.betas = reshape(model.betas,[nh nConds nVoxels]);
model.stdevs = reshape(model.stdevs,[nh nConds nVoxels]);
model.sems = reshape(model.sems,[nh nConds nVoxels]);


return
