function [newDt, newScan] = applyGlm(view, scans, params, newDt, newScan);
%
% [newDt, newScan] = applyGlm([view], [scans], [params], [newDt], [newScan]);
%
% Apply a General Linear Model (GLM) to a mrVista view,
% saving the results for further analyses in a separate
% data type.
%
% view: defaults to selected inplane if omitted.
%
% scans: scans from which to take data for GLM. Defaults
% to scan group assigned to view's current scan.
%
% params: event-related analysis params. See er_getParams,
% er_defaultParams. 
%
% newDt: name of the data type in which to save the results.
% Defaults to 'GLMs'. Makes the new dt if it didn't exist
% before. (Note: it may be possible to keep everything in 
% an existing data type, like 'Original', since this analysis
% doesn't create new tSeries, but I have to check.)
%
% newScan: scan # in which to store results. Defaults to 
% appending a new one on to the selected new data type.
%
% Returns the name and # of the new data type and scan in
% which the data are saved.
% 
% ras 06/06.
% ras, 02/06: fixed bug w/ 'newDt' vs. 'newDt'.
mrGlobals;
tic;

if ieNotDefined('view')
    view = getSelectedInplane;
    if isempty(view), help(mfilename); return; end
end

if ieNotDefined('scans')
    [scans srcDt] = er_getScanGroup(view);
    view = selectDataType(view,srcDt);
end

if ieNotDefined('params'), params = er_getParams(view); end
if ieNotDefined('newDt'), newDt = 'GLMs'; end

% ensure the target data type exists
srcDt = viewGet(view,'curDataType');
tgtDt = existDataType(newDt);
if tgtDt==0
    % data type doesn't currently exist, so we'll
    % make it:
    tgtDt = length(dataTYPES)+1;
    dataTYPES(tgtDt).name = newDt;
end

% now, check if we need the default new scan # (append)
if ieNotDefined('newScan')
    newScan = length(dataTYPES(tgtDt).scanParams)+1;
end

% update dataTYPES
dataTYPES(tgtDt).scanParams(newScan) = ...
    dataTYPES(srcDt).scanParams(scans(1));
dataTYPES(tgtDt).scanParams(newScan).annotation = newDt;
dataTYPES(tgtDt).scanParams(newScan).scanGroup = newDt;
dataTYPES(tgtDt).blockedAnalysisParams(newScan) = ...
    dataTYPES(srcDt).blockedAnalysisParams(scans(1));
if isempty(dataTYPES(tgtDt).eventAnalysisParams)
    % initialize
    dataTYPES(tgtDt).eventAnalysisParams = params;
else
    dataTYPES(tgtDt).eventAnalysisParams(newScan) = params;
end

% default name (annotation) of the new scan 
names = {dataTYPES.name};
newDt = sprintf('%s: %s',names{srcDt},num2str(scans));

% set up a wait bar (a little higher than other waitbars
% which will appear when running GLM on each slice)
msg = sprintf('Applying GLM to %s...',newDt);
hwait = waitbar(0,msg);
pos = get(hwait,'Position');
pos(2) = pos(2) + 1.2*pos(4);
set(hwait,'Position',pos);

% put some informative text in command window
fprintf('\n\n\t****** Applying GLM: %s ******\n',newDt);
fprintf('\t(This will be saved in %s data type scan %i)\n',...
        names{tgtDt},newScan);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   ***** Main Loop:  *****                                 %
% Go through each slice, load data, apply GLM, save results %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
nSlices = viewGet(view,'numSlices');
for slice = 1:nSlices
    model = applyGlmSlice(view,slice,scans,params);
    saveGlmSlice(view,model,newDt,newScan,params);
    	
	% save the updated session params
	saveSession;
    
    waitbar(slice/nSlices,hwait);
end

close(hwait);

% select the GLM data type
view = selectDataType(view, newDt);

% re-select the GLM data type (update the scan slider, etc)
view = selectDataType(view,newDt);
view = setCurScan(view,newScan);

fprintf('\n\n\t****** GLM Done. Time: %5.0g min %2.2g sec ******.\n\n',toc/60,mod(toc,60));


return

    