function map = mrvColorMaps(name,nColors);
%
% map = mrvColorMaps(name,[nColors]);
%
% Color maps used in mrVista.
%
% list = mrvColorMaps without input arguments
% returns a list of the currently-defined
% color maps.
%
% map = mrvColorMaps([name or num]) returns a 3x256
% color map corresponding to the name, or its index
% number in the list of all colormaps (useful for
% popup menus etc).
%
% map = mrvColorMaps('user') prompts for a user-defined
% colormap. This can be specified in one of two ways:
%   (1) Enter the # of colors to use. A prompt pops up
%       with a GUI to edit manually each of the colors.
%       You can also invoke the matlab CMAPEDITOR on this 
%       for further options.
%
%   (2) Save a matlab file which contains the variable 
%       'cmap'. At the dialog, leave the # of colors empty.
%       A second dialog will appear prompting for this file.
%
% map = mrvColorMaps([name or num], nColors) returns a 
% 3xnColors preset color map.
%
% ras 07/05.
% ras 10/05: added support for external color maps.
% ras 11/05: added to mrVista 1.0 toolbox, from mrVista 2.0, b/c
% I've found it useful.
if nargin==0
    % output a list of names
    map = {'hot' 'cool' 'jet' 'hsv' 'gray' 'autumn' 'winter'...
        'red binary' 'green binary' 'blue binary' ...
        'red' 'green' 'blue' 'redgreen' 'user'};
    return
end

if ~exist('nColors','var') | isempty(nColors), nColors = 256; end

if isnumeric(name)
    % convert from index # into name
    list =  mrvColorMaps;
    name = list{name};
end

map = zeros(nColors,3);
switch lower(name)
    case 'hot', map = hot(nColors);
    case 'cool', map = cool(nColors);
    case 'jet', map = jet(nColors);
    case 'hsv', map = hsv(nColors);
    case 'gray', map = gray(nColors);
    case 'autumn', map = autumn(nColors);
    case 'winter', map = winter(nColors);
    case 'red binary', map(:,1) = 1;
    case 'green binary', map(:,2) = 1;
    case 'blue binary', map(:,3) = 1;
    case 'red', map(:,1) = linspace(0,1,nColors);
    case 'green', map(:,2) = linspace(0,1,nColors);
    case 'blue', map(:,3) = linspace(0,1,nColors);
    case 'redgreen',
        map(:,1) = linspace(0.2,1,nColors);
        map(:,2) = linspace(1,0.2,nColors);
    case 'user', map = mrvUserColormap;
end

return
% /-------------------------------------------------------------------/ %



% /-------------------------------------------------------------------/ %
function map = mrvUserColormap;
% Use the colormap editor to get a colormap
%
% map = mrvUserColormap;
%
% ras 10/2005.
q='Enter # of colors in new colormap (leave empty to load from file)';
resp = inputdlg(q,'mrvColormaps',1,{'256'});
if isempty(resp{1})
    % load from file
    p = 'Select a .mat file with the colormap as a ''cmap'' variable.';
    [f pth] = uigetfile('*.mat',p);
    try
        load(fullfile(pth,f),'cmap');
        map = cmap;
    catch
        msg='Can''t find file w/ cmap variable, returning empty cmap.';
        warning(msg); map=[]; return
    end
else
    nColors = str2num(resp{1});
    map = cmapEdit(hot(nColors));
end

return
