function view = bicolorCmap(view);
%
% view = bicolorCmap(view);
% 
% Makes colormap array with:
%   gray scale - 1:numGrays
%   winter colors - values in which map < 0
%   black - values in which map=0
%   autumn colors - values in which map > 0
%
% This is useful for plotting contrast maps in which both
% positive and negative effects are displayed (for related updates, see
% loadParameterMap, computeContrastMap).
%
% ras, 03/04, off of hotCmap
% ras, 05/05, now takes view as an input arg rather
% than a map, and returns a view as well.
numGrays  = view.ui.mapMode.numGrays;
numColors = view.ui.mapMode.numColors;

% get map for current scan
scan = getCurScan(view);
if ~isempty(view.map) & ~isempty(view.map{scan})
    map = view.map{scan};
    
    % we have a map on which we want to base
	% the bicolor cmap. Find out whether the
	% map only contains positive or negative 
	% values, or if it crosses zero, and set
	% the map appropriately:
	hi = max(map(:));
	lo = min(map(:));
	
	rng = linspace(lo,hi,numColors);
	
	if lo >= 0 % all positive
        colors = autumn(numColors);
	elseif hi < 0 % all negative 
        colors = flipud(winter(numColors));
	else        % crosses zero
        colors = zeros(numColors,3);
        neg = length(find(rng < 0));
        colors(neg,:) = [0 0 0]; % black when crosses
        colors(1:neg-1,:) = flipud(winter(neg-1));
        colors(neg+1:end,:) = autumn(numColors-neg);
	end
    
    clipMode = [lo hi];

    % let's go ahead and map the absolute value
	% of the map onto the 'co' field for this scan:
	% this only works for parameter maps, but that may 
	% be all we care about at this point:
	view.co{scan} = normalize(abs(map));
    
else
    
    % no map to access, just make
    % a color map w/ autumn and winter 
    % split along the middle:
    colors = flipud(winter(numColors/2));
    colors = [colors; autumn(numColors-size(colors,1))];
    clipMode = 'auto';
    
end

% combine grayscale / color parts of map
cmap = [gray(numGrays);colors];

% set the param map field appropraitely (this is locked
% in right now, so it may be difficult to do this
% for co/amp/ph modes, but I doubt that'd be needed).
view.ui.mapMode.cmap = cmap;
view.ui.mapMode.name = 'bicolorCmap';
view.ui.mapMode.clipMode = clipMode;


return
