function rmPlot(view,wplot,modelId);
% rmPlot - plot retinotopic model voxel results
%
% rmPlot(view,wplot,modelId);
% 

% 2006/02 SOD: wrote it.

if ieNotDefined('view'),    error('Need view struct'); end;
if ieNotDefined('wplot'),   wplot = 'all';             end;
if ieNotDefined('modelId'), modelId = [];              end;

% load file with data
rmFile = viewGet(view,'rmFile');
if isempty(rmFile),
  disp(sprintf('[%s]:No file selected',mfilename));
  return;
end;
load(rmFile,'model','params');

% get model id 
if isempty(modelId),
  % get model names
  for n=1:length(model),
    modelNames{n} = rmGet(model{n},'desc');
  end;
  modelId = menu('Select stimulus type: ', modelNames);
end;

% get time series and roi-coords
[tSeries,coords] = gettimeseries(view,params);

% make trends 
[trends, ntrends] = rmMakeTrends(params);
trendID = ntrends;

% load rss
rss = getrfprofile(view,model{modelId},coords,'rss');

% make on/off if needed
switch modelNames{modelId},
 case {'on/off only fit'},
  stim    = getonoffsequence(params);
  trendID = trendID + 1;
  betaID  = getbetas(view,params,model{modelId},coords,trendID);
  for n=1:size(coords,2),
    prediction(:,n) = [stim trends]*betaID(n,:)';
  end;
  RF      = mean(betaID(:,1));
  pred    = mean(prediction,2);
  
 case {'2D RF (x,y,sigma) fit (positive only)',...
       '2D RF (x,y,sigma) fit (no constraint)'},
  [RFs, pred] = getrfsequence(view,params,model{modelId},coords);
  trendID     = trendID + 1;
  betaID      = getbetas(view,params,model{modelId},coords,trendID); 
  for n=1:size(coords,2),
    prediction(:,n) = [pred(:,n) trends]*betaID(n,:)';  
    RFs(:,n)        = RFs(:,n).*betaID(n,1);
  end;
  pred = mean(prediction,2);
  RF   = mean(RFs,2);
 
 case {'2D RF (x,y,sigma) with on/off fit (positive only)',...
       '2D RF (x,y,sigma) with on/off fit (no constraint)'},
  stim        = getonoffsequence(params);
  [RFs, pred] = getrfsequence(view,params,model{modelId},coords);
  trendID     = trendID + 2;
  betaID      = getbetas(view,params,model{modelId},coords,trendID);
  for n=1:size(coords,2),
    prediction1(:,n) = [pred(:,n) stim trends]*betaID(n,:)';
    prediction2(:,n) = [          stim trends]*betaID(n,2:end)';
    RFs(:,n)   = RFs(:,n).*betaID(n,1);
    RFs(:,n)   = RFs(:,n) + RFs(:,n).*0+betaID(n,2);
  end;
  pred(:,1)   = mean(prediction1,2);
  pred(:,2)   = mean(prediction2,2);
  RF          = mean(RFs,2);
  
 case {'Double 2D RF (x,y,sigma) fit (no constraint)'},
  stim        = getonoffsequence(params);
  [RFs, pred, RFs2, pred2] = getrfsequence(view,params,model{modelId},coords);
  trendID     = trendID + 2;
  betaID      = getbetas(view,params,model{modelId},coords,trendID);
  for n=1:size(coords,2),
    prediction1(:,n) = [pred(:,n) pred2(:,n) trends]*betaID(n,:)';
    prediction2(:,n) = [          pred2(:,n) trends]*betaID(n,2:end)';
    RFs(:,n)   = RFs(:,n).*betaID(n,1);
    RFs(:,n)   = RFs(:,n) + RFs2(:,n).*betaID(n,2);
  end;
  pred(:,1)   = mean(prediction1,2);
  pred(:,2)   = mean(prediction2,2);
  RF          = mean(RFs,2);
  
 otherwise,
  error(sprintf('Unknown modelName: %s',modelNames{modelId}));
end;  

% now plot
switch lower(wplot),
 case {'rf'},
  rfPlot(params,RF);
  
 case {'ts'},
  tsPlot(tSeries,pred,rss,params,modelNames{modelId});
  
 case {'all'},
  rfPlot(params,RF);
  tsPlot(tSeries,pred,rss,params,modelNames{modelId});
%  disp(sprintf('[RF]: x,y coordinates: %.4f,%.4f degrees',...
%               mean(rfId(4.:)), mean(rfId(5.:))));
%  disp(sprintf('[RF]: sigma: %.4f,%.4f degrees',...
%               mean(rfId(4.:)), mean(rfId(5.:))));
  
 otherwise,
  error(sprintf('[%s]: unknown value of wplot: %s',mfilename, ...
                wplot));
end;
return;

%--------------------------------------
function betaID = getbetas(view,params,model,coords,trendID);
for n=1:max(trendID),
  betaID(:,n) = getrfprofile(view,model,coords,...
                                  sprintf('bcomp%d',n));
end;
return;
%--------------------------------------

%--------------------------------------
function [RFs, pred, RFs2, pred2] = getrfsequence(view,params,model,coords);
% get RF parameters
rfId(:,1) = getrfprofile(view,model,coords,'sigmamajor');
rfId(:,2) = getrfprofile(view,model,coords,'sigmaminor');
rfId(:,3) = getrfprofile(view,model,coords,'sigmatheta');
rfId(:,4) = getrfprofile(view,model,coords,'x0');
rfId(:,5) = getrfprofile(view,model,coords,'y0');

for n=1:size(coords,2),
  RFs(:,n)   = rfGaussian2d(params.analysis.X,params.analysis.Y,...
                            rfId(1),rfId(2),rfId(3),rfId(4),rfId(5));
  pred(:,n)  = rfMakePrediction(params,rfId);
end;
s1 = mean(rfId(:,1));

if nargout > 2,
  rfId(:,1) = getrfprofile(view,model,coords,'sigma2major');
  rfId(:,2) = getrfprofile(view,model,coords,'sigma2minor');
  rfId(:,3) = getrfprofile(view,model,coords,'sigma2theta');
  rfId(:,4) = getrfprofile(view,model,coords,'x0');
  rfId(:,5) = getrfprofile(view,model,coords,'y0');

  disp(sprintf('[%s]: x,y,s1(M),s2(M) = %.4f, %.4f, %.4f, %.4f (deg)',...
               mfilename,...
               mean(rfId(:,4)),mean(rfId(:,5)),s1,mean(rfId(:,1))))
  for n=1:size(coords,2),
    RFs2(:,n)   = rfGaussian2d(params.analysis.X,params.analysis.Y,...
                               rfId(1),rfId(2),rfId(3),rfId(4),rfId(5));
    pred2(:,n)  = rfMakePrediction(params,rfId);
  end;
else,
  disp(sprintf('[%s]: x,y,s(M) = %.4f, %.4f, %.4f',...
               mfilename,...
               mean(rfId(:,4)),mean(rfId(:,5)),mean(rfId(:,1))))
end;
return;
%--------------------------------------


%--------------------------------------
function [stim] = getonoffsequence(params);
% make stim on/off sequence
stim = [];
for ii = 1:length(params.stim),
  stim = [stim; params.stim(ii).stimOnOffSeq];
end;
return
%--------------------------------------


%--------------------------------------
function [ts, coords] = gettimeseries(view,params);
params.wData = 'roi';
[tSeries, params] = rmLoadData(view,params);
ts     = mean(tSeries,2);
coords = params.coords;
switch lower(view.viewType),
 case 'inplane'
  rsFactor = upSampleFactor(view,1);
  if length(rsFactor)==1
    coords(1:2,:) = round(coords(1:2,:)/rsFactor(1));
  else
    coords(1,:) = round(coords(1,:)/rsFactor(1));
    coords(2,:) = round(coords(2,:)/rsFactor(2));
  end;
  coords = unique(coords','rows')';
  
 case {'gray'}
  coordsInd    = zeros(1,size(coords,2));
  allcoords    = viewGet(view,'coords');
  % loop because intersect orders the output
  for n=1:size(coords,2);
    [tmp, coordsInd(n)] = intersectCols(allcoords,coords(:,n));
  end;
  coords = coordsInd;
  
 otherwise
  error(sprintf('[%s]:unknown viewType %s',...
                mfilename,viewGet(view,'viewType')));
end;
return;
%--------------------------------------

%--------------------------------------
function rfp = getrfprofile(view,model,coords,param);
tmp = rmGet(model,param);
rfp = zeros(size(coords,2),1);
switch lower(view.viewType),
 case 'inplane'
  for n=1:length(rfp),
    rfp(n) = tmp(coords(1,n),coords(2,n),coords(3,n));
  end;
  
 case 'gray'
  rfp = tmp(coords);
end;
return;
%--------------------------------------

%--------------------------------------
function tsPlot(ts,fit,rss,params,modelName);
figure;
x=[1:length(ts)].*params.stim(1).framePeriod;
nscans = length(params.stim);
sepx = [1:nscans-1].*max(x)./nscans;

subplot(2,1,1);
plot(x,ts,'bo:');hold on;
plot(x,fit(:,1),'r');
if size(fit,2) > 1,
  plot(x,fit(:,2),'g--');
end;
h = axis;
axis([min(x) max(x) h(3) h(4)]);
for n=1:length(sepx),
  plot([1 1].*sepx(n),[h(3) h(4)],'k:');
end;
title(sprintf('Model: %s',modelName));
xlabel('time (sec)');
ylabel('% BOLD signal change');

subplot(2,1,2);
plot(x,ts-fit(:,1),'b');hold on;
plot(x,fit(:,1),'r');
title(sprintf('Residuals (RSS: %f)',mean(rss)));
xlabel('time (sec)');
ylabel('% BOLD signal change');

return;
%--------------------------------------
