function [ctics,hd_resp]=rfConvolveTC(tics,FrameTime,wHrf,hrfParams);
% rfConvolveTC - convolve a given sequence with hdrf
% [ctics]=rfConvolveTC(tics,frametime,wHrf,hrfParams);
%    Convolve a given sequence with a haemodynamic 
%    response function (wHrf) with params (hrfParams)
%   
%    inputs : tics (sequence to be convolved)
%             FrameTime (TR, secs)
%             wHrf      ('boynton','twogammas') 
%             hrfParams ([Torr Td],[peak1 fwhm1 peak2 fwhm2 dip])
%
%    'boynton' defaults: [1.68 3] 
%    From Boynton et al, J Neurosci (1996) 16:4207-4221.
%    'twogammas' defaults:  [5.4 5.2 10.8 7.35 0.35]
%    From Glover, NeuroIm, (1999) 9:416-429, also used by
%    Worsley's fmristat: Worsley et al. NeuroIm (2002) 15:1-15.
%

% SOD: wrote it.


if nargin < 3,
  help(mfilename);
  return;
end;
if nargin < 4,
   hrfParams = [];
end;

% if wHrf is char then make hrf otherwise we assume that wHrf
% provides the hd_resp (ie we only need to make it once)
if ischar(wHrf)
  % make hemodynamic response
  FIR_lengthsec = 50;
  FIR_frames    = ceil(FIR_lengthsec/FrameTime);
  FIR_time      = [0:FIR_frames]*FrameTime;
  switch lower(wHrf)
   case {'b','boynton'}
    hd_resp    = boynton(FIR_time,hrfParams);
   case {'t','g','gamma','twogammas'}
    hd_resp    = twogammas(FIR_time,hrfParams);
   otherwise
    disp(sprintf('[%s]:Unknown hrf function: %s',wHrf));
    help(mfilename);
    return;
  end;
else,
  hd_resp = wHrf;
end;
  
% convolve stimulus sequence with hdrf
ctics = zeros(size(tics));
for n=1:size(ctics,2),
  tmp = conv(tics(:,n)',hd_resp);
  tmp = tmp(1:size(tics,1))';
  ctics(:,n) = tmp;
end;

%-------------------------------------------------------------
function [h]=boynton(t,params);
% t : a range of latencies
% params(1) : the time constant (sec)
% params(2) : the phase delay
% from Boynton et al, J. Neurosci July 1 1996. 16(13):4207-4221
% they used  T=1.48->1.68, n=3
 
if nargin<2 | isempty(params),
  params=[1.68 3];
end;

h = zeros(size(t));
for ii=1:length(t),
  h(ii)=( (t(ii)/params(1))^(params(2)-1) )*exp(-t(ii)/params(1));
  h(ii)=h(ii)/(params(1)*gamma(params(2))); % gamma (n) is same as (n-1)!
end;
 
h=h./sum(h); %  normalize 
 
return;

%-------------------------------------------------------------
function [h]=twogammas(t,params)
% t: a range of latencies
% params(1): peak gamma 1
% params(2): fwhm gamma 1
% params(3): peak gamma 2
% params(4): fwhm gamma 2
% params(5): dip
% Final hrf is:   gamma1/max(gamma1)-dip*gamma2/max(gamma2)
% from Glover, NeuroImage, 9:416-429

if nargin < 2 | isempty(params),
  params = [5.4 5.2 10.8 7.35 0.35];
end;

% params
peak1 = params(1);
fwhm1 = params(2);
peak2 = params(3);
fwhm2 = params(4);
dip   = params(5);

% sanity check
if(peak1 == 0 | fwhm1 ==0),
  disp(sprintf('[%s]: zero params'));
  params,
  return;
end;
  
% Taylor:
alpha1=peak1^2/fwhm1^2*8*log(2);
beta1=fwhm1^2/peak1/8/log(2);
gamma1=(t/peak1).^alpha1.*exp(-(t-peak1)./beta1);

if peak2>0 & fwhm2>0
  alpha2=peak2^2/fwhm2^2*8*log(2);
  beta2=fwhm2^2/peak2/8/log(2);
  gamma2=(t/peak2).^alpha2.*exp(-(t-peak2)./beta2);
else 
  gamma2=min(abs(t-peak2))==abs(t-peak2);
end
h = gamma1-dip*gamma2;
h = h./sum(h);

return;
