function [view, M] = betweenScanMotComp(view, newView, baseScan, targetScans)
%
% [view, M] = betweenScanMotComp(view, baseScan, targetScans)
%
% Robust 3D rigid body motion compensation between MEAN MAPS of different scans
%
% If you change this function make parallel changes in:
%    motionComp, inplaneMotionComp
%
% 06/06/00 - ON
% Ress, 06/05. Added some feedback on motion magnitude, and set 0.15 voxel
% threshold to perform correction. Removed "copy scan" function because we
% now make a new dataType.

mrGlobals

% Get or compute Mean Maps.
view = loadMeanMap(view);
meanMap = view.map;

% if the number of slices is too small, repeat the first and last slice
% to avoid running out of data (the derivative computation discards the
% borders in z, typically 2 slices at the begining and 2 more at the end)
if size(meanMap,3)<=8
  meanMap = cat(3, meanMap(:,:,1,:), meanMap(:,:,1,:), meanMap,...
    meanMap(:,:,end,:), meanMap(:,:,end,:));
end

% get base mean map
baseMeanMap = meanMap{baseScan};

ndataType = newView.curDataType;

% Do motion estimation/compensation for each scan.
for iScan=1:length(targetScans)
  scan = targetScans(iScan);
  slices = sliceList(view,scan);
  nSlices = length(slices);
  nFrames = numFrames(view,scan);
  dims = sliceDims(view,scan);
  
  % Load tSeries from all slices into one big matrix 
  volSeries = zeros([dims(1) dims(2) nSlices nFrames]);
  waitHandle = waitbar(0,'Loading tSeries from all slices.  Please wait...');
  for slice=slices
    waitbar(slice/nSlices);
    ts = loadtSeries(view,scan,slice);
    for frame=1:nFrames
      volSeries(:, :, slice, frame) = reshape(ts(frame,:),dims);
    end
  end
  close(waitHandle)
  
  if scan == baseScan
    totalMot = 0;
  else
    % estimate motion between mean maps
    M = estMotionIter3(baseMeanMap,meanMap{scan},3,eye(4),1,1); % rigid body, ROBUST
    midX = [dims/2 nSlices/2]';
    midXp = M(1:3, 1:3) * midX; % Rotational motion
    rotMot = sqrt(sum((midXp - midX).^2));
    transMot = sqrt(sum(M(1:3, 4).^2)); % Translational motion
    totalMot = sqrt(rotMot^2 + transMot^2);
    disp(['Scan ', int2str(scan), ' - motion (voxels): rot = ', num2str(rotMot), '; trans = ', num2str(transMot), ...
        ' total = ', num2str(totalMot)])
  end

  if totalMot > 0.15
    % compute the warped volume series according to the previously computed M
    waitHandle = waitbar(0,['Warping scan ', num2str(scan),'...']);
    for frame = 1:nFrames
      waitbar(frame/nFrames)
      % warp the volume putting an edge of 1 voxel around to avoid lost data
      volSeries(:,:,:,frame) = warpAffine3(volSeries(:,:,:,frame), M, NaN, 1);
    end
    close(waitHandle)
  end % motion correct
    
  % Save tSeries
  tSeries = zeros(size(ts));
  numPixels = size(tSeries,2);
  waitHandle = waitbar(0,'Saving tSeries...');
  for slice=slices
    waitbar(slice/nSlices);
    for frame=1:nFrames
      tSeries(frame, :) = reshape(volSeries(:,:,slice,frame), [1 numPixels]);
    end
    savetSeries(tSeries,newView,scan,slice);
  end
  close(waitHandle)
  clear volSeries
  dataTYPES(ndataType).scanParams(scan) = dataTYPES(view.curDataType).scanParams(scan);
  dataTYPES(ndataType).blockedAnalysisParams(scan) = dataTYPES(view.curDataType).blockedAnalysisParams(scan);
  dataTYPES(ndataType).eventAnalysisParams(scan) = dataTYPES(view.curDataType).eventAnalysisParams(scan);
  dataTYPES(ndataType).scanParams(scan).annotation = dataTYPES(view.curDataType).scanParams(scan).annotation;
end % scan LOOP
  
return
