function varargout = vAnatomyfromIMA(anatdir, outdir)

% produces vAnatomy from sagital MPRAGE *.IMA files


if ~isdir(anatdir)
    error( ['Directory ', anatdir ' does not exist'])
end

list = readIMAFileListing( anatdir, 'min' , 'fullname');
if isempty(list)
    fprintf('No IMA files found in ''%s''\n', anatdir)
    return
end

% determine output file
if nargin > 1 & isdir(outdir)
    outfile = fullfile( outdir, 'vAnatomy.dat' );
else
    outdir = list(1).Path;
    outfile = fullfile( list(1).Path, 'vAnatomy.dat' );
end

fprintf('%s: output directory: ''%s''\n', mfilename, outdir)

if exist(outfile)
    rep = questdlg('vAnatomy.dat already exists.  Overwrite?', 'Warning');
    if isempty( strmatch(lower(rep), 'yes') )
        return
    end
end


hdr = dicominfo( list(1).name );
anat = zeros( hdr.Columns, hdr.Rows, length(list));


%warning('Left-right orientation not definite! CHECK')
fprintf('%s: Processing ', mfilename)
h = waitbar(0, 'Generating vAnatomy.dat');

for i = 1:length(list)
    if rem( floor(i/4), 2)
        fprintf('\b')
    else 
        fprintf('.')
    end
    anat(:,:,i) = dicomread( list(i).name )';
    waitbar(i/length(list), h);
end

% scale over full range - added 21st Jan 2004
fprintf('\nvAnatomyfromIMA: Scaling data\n');

perc=0; fullout = 0;
% here we want to probably use the 2 & 98 % percentiles to increase the
% dynamic range and minimise the inflow artefact effect
if perc
    
    disp('vAnatomyfromIMA: calculating percentiles')
    
    lims = percentile(anat, [0.02, 0.99999]); % v.conservative
    
    % attempting to scale by reducing the max by say 10%
    %tmp = max(anat(:));
    %lims(2) = tmp*0.9
    
    lowerLimSubs = anat < lims(1);
    upperLimSubs = anat > lims(2);
    anat( lowerLimSubs ) = lims(1);
    anat(  upperLimSubs ) = lims(2);
end

% scale 0->255
anat = bytescl(anat);

fprintf('\n%s: Writing file. \n', mfilename)

% save to .dat
fid = fopen(outfile, 'w');
str = sprintf('rows %f (%f mm/pixel)\ncols %f (%f mm/pixel)\nplanes %f (%f mm/pixel)\n*\n', ...
                    hdr.Rows, hdr.PixelSpacing(1), hdr.Columns, hdr.PixelSpacing(2),  length(list), hdr.SliceThickness);

fwrite(fid, str, 'char');
fwrite(fid, uint8(anat), 'uint8');
fclose(fid);

% save indices of scaled pixels to .dat files?
fullout=0;
if fullout
    fid = fopen( fullfile(outdir, 'vAnatomyLower.dat'), 'w');
    fwrite(fid, str, 'char');
    fwrite(fid, uint8(lowerLimSubs), 'uint8');
    fclose(fid);
    fid = fopen( fullfile(outdir, 'vAnatomyUpper.dat'), 'w');
    fwrite(fid, str, 'char');
    fwrite(fid, uint8(upperLimSubs), 'uint8');
    fclose(fid);
end


% write README file
fprintf('%s: Writing README_anat.txt\n', mfilename);
fidR = fopen( fullfile(outdir, 'README_anat.txt'), 'wt' );
fprintf(fidR, 'Aq date: %s\n', hdr.AcquisitionDate);
fprintf(fidR, 'Subject: %s\n', hdr.PatientID);
fprintf(fidR, 'Age     : %s\n\n', hdr.PatientsAge);



try
fprintf(fidR, 'Magnet: %s %s (%.2f T)\n', hdr.Manufacturer, hdr.ManufacturersModelName, hdr.MagneticFieldStrength);
end
try
fprintf(fidR, 'Imaging sequence:   %s (%s)\n\n', hdr.SeriesDescription, hdr.SequenceName);
end
try
fprintf(fidR, 'Resolution (mm)   :   %f x %f x %f (inplane x slice thickness N.B. not necesarily x,y,z)\n', hdr.PixelSpacing, hdr.SliceThickness);
end
try
fprintf(fidR, 'Acquisition Matrix:   %d x %d x %d\n', hdr.Rows, hdr.Columns, length(list));
end
fclose(fidR);

% delete waitbar
close(h);

if nargout 
    varargout{1} = anat;
end