
function reslicedImg = resliceVoxles(imgCubeData, rowSpan, colSpan, planeSpan)
% reslicedImg = resliceVoxles(imgCubeData, nRows, nCols, nPlanes, rowSpan, colSpan, planeSpan)
% Generalized function to reslice the voxels in a cubic array of image data so that voxels
% have are cubic in the physical space.
%
% Each entry of the matrix corresponds to one voxel in the image data.
% rowSpan, colSpan, and PlaneSpan refer to the linear distance spanned (in anatomical space) by each
% of the dimensions in the image data. For example if the image data had a field of view of 240mm in the
% x and y dimensions and a f.o.v. of 148mm in the z dimension, then
%  rowSpan = colSpan = 240
%  planeSpan = 148
% 
% resliceVoxels() returns a 3D matrix of voxel values.

error(nargchk(4,4,nargin));

disp('Reslicing image to cubic voxels...');

[nRows,nCols,nPlanes] = size(imgCubeData);


disp('Determining voxel dimensions...');

mmPerVoxelInX = colSpan / nCols;
disp([num2str(mmPerVoxelInX) 'mm per voxel in the x axis.']);

mmPerVoxelInY = rowSpan / nRows;
disp([num2str(mmPerVoxelInX) 'mm per voxel in the y axis.']);

mmPerVoxelInZ = planeSpan / nPlanes;
disp([num2str(mmPerVoxelInZ) 'mm per voxel in the z axis.']);

%%
% Compute the target cubic dimension by finding the smallest dimension

targetDimension = min([mmPerVoxelInX mmPerVoxelInY mmPerVoxelInZ]);
disp(['All voxels will be resliced to be cubic, ' num2str(targetDimension) 'mm per voxel.']);

% Compute number of slices in each dimension (rounded to the nearest integer).
numResampledSlicesX = round(colSpan / targetDimension);
numResampledSlicesY = round(rowSpan / targetDimension);
numResampledSlicesZ = round(planeSpan / targetDimension);


% Reslice the image. We do this in one step using inter3:

disp(['Resampling the x axis to ' int2str(numResampledSlicesX) ' slices...']);
disp(['Resampling the y axis to ' int2str(numResampledSlicesY) ' slices...']);
disp(['Resampling the z axis to ' int2str(numResampledSlicesZ) ' slices...']);

reslicedImg = zeros(numResampledSlicesX, numResampledSlicesY, numResampledSlicesZ);
reslicedImg = interp3(imgCubeData, linspace(1,nCols,numResampledSlicesX),linspace(1,nRows,numResampledSlicesY)',linspace(1,nPlanes,numResampledSlicesZ)');

disp(['Volume dimensions are now: ' int2str(size(reslicedImg))]);

return;