function [brainMask,checkSlices] = mrAnatExtractBrain(img, mmPerVox, betLevel)
%
% [brainMask,checkSlices] = mrAnatExtractBrain(img, [mmPerVox=[1,1,1]], [betLevel=0.5])
%
% Uses the FSL BET tool to compute a brain mask for the given image volume.
% The betLevel is the BET fractional intensity threshold parameter. It
% should be [0-1]. The default of 0.5 usually works well. Smaller values
% will yield a larger brain estimate. If you get brain stuff chopped off,
% try a lower value. If you get non-brain stuff included, try a larger
% value.
%
% If you provide an array of betLevel values, you'll get a cell array of
% length(betLevel) brain masks back.
%
% checkSlices is a montage of every 3rd slice with the mask shown in blue
% and the underlying anatomy on yellow. Again, for length(betLevels)>1,
% you'll get a cell array.
%
% WARNING: this currently only runs on linux!
%
%
% HISTORY:
% 2006.02.02 RFD: wrote it.

if(~exist('betLevel','var') | isempty(betLevel))
    betLevel = 0.5;
end
if(~exist('mmPerVox','var') | isempty(mmPerVox))
    mmPerVox = [1 1 1];
end

%bet = '/usr/local/fsl-3.2/bin/bet2';
bet = fullfile(fileparts(which(mfilename)), 'bet2');
out = fullfile(tempdir,'bet_tmp');
if(ischar(img))
    out = img;
else
    mx = max(abs(img(:)));
    img = int16(img./mx.*32767+0.5);
    saveAnalyze(img, out, mmPerVox);
end
% We could specify a better starting position for the BET surface
% sphere estiamte, e.g., by using the talairach landmarks. 
betOut = [out '_brain'];
for(ii=1:length(betLevel))
    betCmd = [bet ' ' out ' ' betOut ' -mnf ' num2str(betLevel(ii))];
    unix(['export FSLOUTPUTTYPE=ANALYZE ; ' betCmd]);
    brainMask{ii} = logical(loadAnalyze([betOut '_mask']));
    if(nargout>1)
        slices = [1:3:size(img,3)-6];
        im = double(makeMontage(img, slices));
        im = mrAnatHistogramClip(im, 0.4, 0.99);
        im = uint8(im./max(im(:)).*255+0.5);
        im(:,:,2) = im(:,:,1);
        im(:,:,3) = uint8(double(makeMontage(brainMask{ii},slices)).*255);
        checkSlices{ii} = im;
    end
end
if(length(betLevel)==1)
    brainMask = brainMask{1};
    if(nargout>1) checkSlices = checkSlices{1}; end
end
        
return