function class = readClassFile(filename,headerOnlyFlag,voiOnlyFlag);
% 
%  class = readClassFile(filename,[headerOnlyFlag],[voiOnlyFlag]);
% 
% AUTHOR:  Wandell
% DATE: 06.24.96
% PURPOSE: 
%   Read the classification file produced by mrGray.  
%   The results are returned in a structure.
% 
% Examples:
%    Read in all the data, not just the part in the VOI
%    class = readClassFile('rightWhole.class',0,0);
%
%    Read in just the header.
%    class = readClassFile('rightWhole.class',1);  
%
% MODIFICATIONS:
% 07.20.98 SJC	commented out figures
% 09.02.98 SJC	Added the optional input argument 'headerOnlyFlag'
%		which, if set to 1, makes the function only read
%		and return the header information instead of
%		reading in the entire class file.
% 04.13.99 SJC Added the optional input argument 'voiOnlyFlag'
%		which, if set to 1, crops the classification data to the
%		volume of interest only, and if set to 0 does not crop it.
% 06.18.99 SJC Changed the pause after the message window pops up to 'drawnow'
% 09.15.99 BW/WP
%               The VOI now returns in Matlab coordinates (1:N), 
%               rather than C coordinates (0:(n-1)).  This is now 
%               consistent with the update in readGrayGraph.
% 2002.11.01 RFD- added '=>uchar' to force fread to return a uchar array.
% Also removed the GUI msgbox.
% 2003.09.23 RFD: added 3rd option for voiOnlyFlag to allow the full
% (unclipped) data to be returned AND preserves the VOI clipping values in
% the header. Note that the stuct returned by this mode may be incompatible
% with writeCLassFIle. However, it is useful for callers who want to do
% their own clipping.

if ~exist('voiOnlyFlag','var')
    disp('Only reading data from the VOI, not the full volume.')
   voiOnlyFlag = 1;
end
if ~exist('headerOnlyFlag','var')
  headerOnlyFlag = 0;
end

msg = sprintf('Reading the classification file %s...\n',filename);
disp(msg);
% h = msgbox(msg,'readClassFile');
% drawnow

% Save the filename used to read the data
% 
class.filename = filename;

% Set up values for different data types
% 
class.type.unknown = (0*16);
class.type.white   = (1*16);
class.type.gray    = (2*16);
class.type.csf     = (3*16);

% Open the file
% 
fp = fopen(class.filename,'r')

% Read header information
% 
class.header.version = fscanf(fp, 'version= %d\n',1);
class.header.minor = fscanf(fp, 'minor= %d\n',1);

class.header.voi(1) = fscanf(fp, 'voi_xmin=%d\n',1);
class.header.voi(2) = fscanf(fp, 'voi_xmax=%d\n',1);
class.header.voi(3) = fscanf(fp, 'voi_ymin=%d\n',1);
class.header.voi(4) = fscanf(fp, 'voi_ymax=%d\n',1);
class.header.voi(5) = fscanf(fp, 'voi_zmin=%d\n',1);
class.header.voi(6) = fscanf(fp, 'voi_zmax=%d\n',1);

%  This converts VOI from C-0 indexing to Matlab 1-indexing
% 
class.header.voi = class.header.voi + 1;

class.header.xsize = fscanf(fp, 'xsize=%d\n',1);
class.header.ysize = fscanf(fp, 'ysize=%d\n',1);
class.header.zsize = fscanf(fp, 'zsize=%d\n',1);

% if the headerOnlyFlag is set to zero, read in the data
%
%if ~headerOnlyFlag
  csf_mean = fscanf(fp, 'csf_mean=%g\n',1);
  gray_mean = fscanf(fp, 'gray_mean=%g\n',1);
  white_mean = fscanf(fp, 'white_mean=%g\n',1);
  stdev = fscanf(fp, 'stdev=%g\n',1);
  confidence = fscanf(fp, 'confidence=%g\n',1);
  smoothness = fscanf(fp, 'smoothness=%d\n',1);
  class.header.params = ...
    [ csf_mean gray_mean white_mean stdev confidence smoothness];

  % Read the entire raw data set
  % 2002.11.01 RFD- added '=>uchar' to force fread to return a uchar array.
  % This is **much** faster. 
  [im, cnt ] = fread(fp,'uchar=>uchar');
  fclose(fp);

  % Reshape the volume
  % 
  class.data = ...
    reshape(im,[class.header.zsize,class.header.ysize,class.header.xsize]);

  % figure(1);
  % imagesc(class.data(:,:,34)),axis image

  % If the voiOnlyFlag is set, extract that portion of the data,
  % shrinking the total size of the data
  % 
  if voiOnlyFlag>0
      class.data = class.data( ...
          (class.header.voi(1):class.header.voi(2)), ...
          (class.header.voi(3):class.header.voi(4)), ...
          (class.header.voi(5):class.header.voi(6)));
  elseif voiOnlyFlag==0
      % If the voiOnlyFlag was not set, the data size may mis-match
      % the voi size.  So, we need to make sure the voi
      % for the returned class is the entire data set
      %
      disp('Setting class VOI to entire data set');
      class.header.voi(1) = 1; class.header.voi(2) = size(class.data,1);
      class.header.voi(3) = 1; class.header.voi(4) = size(class.data,2);
      class.header.voi(5) = 1; class.header.voi(6) = size(class.data,3);
  else
      disp('Data not clipped to VOI, but VOI header preserved.');
  end

end

% if exist('h'), close(h), end

return;


