function [nodes,edges,classData] = mrgGrowGray(classData,numLayers,layer0)
%
%     [nodes,edges,classData] = mrgGrowGray(classData,[numLayers],[layer0])
%
%   GB 01/11/06
%
% This routine calls the mex function grow_gray with the parameters
% specified by the user and loads the class file if necessary
%
% INPUTS :
%     - classData : should be a variable loaded from a class file or a path
%                   string pointing to the class file
%     - numLayers : number of layers to grow the gray matter
%     - layer0 : optional input argument. (Default 0)
%                   - 0 : no white matter included in the gray graph
%                   - 1 : the white boundary included in the gray graph. 
%                         In this case, the layer number of the boundary will be 0. 
%                   - 2 : All the white matter fully connected is included
%                         in the gray graph
%
% OUTPUTS :
%     - nodes : structure containing nodes of the gray matter
%     - edges : structure containing edges of the gray matter
%     - classData : matlab class file structure containing the output
%
% Example:
%  To visualize a mesh with, say, two layers of gray superimposed do this.
%   fName ='X:\anatomy\nakadomari\left\20050901_fixV1\left.Class';
%   [nodes,edges,classData] = mrgGrowGray(fName,2);
%   wm = uint8( (classData.data == classData.type.white) | (classData.data == classData.type.gray));
%   msh = meshColor(meshSmooth(meshBuildFromClass(wm,[1 1 1])));
%   meshVisualize(msh,2);
%
%  To see the gray matter and white matter in a slice, along with their
%  connectivity, use this.  Change the third argument of mrGrowGray to
%  change the type of connectivity you visualize.
%   fName ='X:\anatomy\nakadomari\left\20050901_fixV1\left.Class';
%   [nodes,edges,classData] = mrgGrowGray(fName,2,2);
%   mrgDisplayGrayMatter(nodes,edges,80,[120 140 120 140]);
%

if isstr(classData), classData = readClassFile(classData); end

if ieNotDefined('numLayers'),  numLayers = 3; end

if ieNotDefined('layer0')
    layer0 = 0;
end

voi = classData.header.voi;
data = classData.data;
data(data(:) == classData.type.gray) = classData.type.unknown;

fprintf('Growing %i gray layers...\n',numLayers);
if layer0, fprintf('White matter included in the gray graph...\n'); 
else fprintf('White matter excluded in the gray graph...\n'); end

[nodes,edges] = grow_gray(data,numLayers,voi,layer0);

grayMatter = nodes(:,find(nodes(6,:) ~= 0));
grayMatter(:,max((grayMatter(1,:) <= voi(1)),(grayMatter(1,:)>voi(2)))) = [];
grayMatter(:,max((grayMatter(2,:) <= voi(3)),(grayMatter(1,:)>voi(4)))) = [];
grayMatter(:,max((grayMatter(3,:) <= voi(5)),(grayMatter(1,:)>voi(6)))) = [];
data(sub2ind(size(data),grayMatter(1,:) - voi(1),grayMatter(2,:) - voi(3),grayMatter(3,:) - voi(5))) = classData.type.gray;

classData.data = data;