function img = meshMultiAngle(msh, angles, savePath, cbarFlag);
% Takes a picture of a mesh at multiple camera angles, and saves as a 
% .png in a directory or pastes in a PowerPoint file.
%
% img = meshMultiAngle([mesh], [angles], [save directory or .ppt file], [cbarFlag]);
%
% mesh: mesh structure of which to take pictures. [Defaults to current
%       mesh of selected mrVista gray view]
% angles: struct array specifying the camera angles for each picture.
%       Each entry should have a 'cRot' field which specifies
%       the camera angles (see meshAngle). 
%       Can also specify a vector of integers, with the indices into
%       the saved angles struct (using meshAngle). E.g. [1 3 2 4]
%       will display the 1st, 3rd, 2nd, and 4th angles, in that order.
%       [If omitted, opens a dialog to select angles from the saved angles
%       -- again, see meshAngle.]
% savePath: directory in which to save a .png file of the image, or 
%       else path to a powerpoint file to paste the image. (PowerPoint
%       is Windows only). [If omitted, doesn't save the image.]
% cbarFlag: if 1, will display image in a separate figure and add 
%       a subplot with the colorbar leged at the bottom. [default 0]
%
%
% Returns a montage image of the mesh from all the specified angles.
% 
%
% ras, 10/2005.
if notDefined('msh'), msh = viewGet(getSelectedGray,'mesh'); end
if notDefined('cbarFlag'), cbarFlag = 0; end

if notDefined('angles')
    % put up a dialog -- but first check there are saved angles:
    fileparts(msh.path)
    angleFile = fullfile(fileparts(msh.path),'MeshAngles.mat');
    if ~exist(angleFile,'file')
        myErrorDlg(['Sorry, you need to save some mesh angles first. ' ...
                   'Use the menu Edit | Save Camer Angles | Save in ' ...
                   'the 3D Window, or the meshAngle function.']);
    else
        load(angleFile,'angles');
        dlg(1).fieldName = 'whichAngles';
        dlg(1).style = 'listbox'; 
        dlg(1).string = 'Take a picture at which camera angles?';
        dlg(1).list = {angles.name};
        dlg(1).value = 1;
        
        dlg(2).fieldName = 'order';
        dlg(2).style = 'edit';
        dlg(2).string = 'OPTIONAL: Order of angles (e.g. [1 2 3 4] vs. [4 3 2 1])?';
        dlg(2).value = ''
        
        resp = generaldlg2(dlg,'Mesh Multi-Angle');
        if isempty(resp), return; end   % user canceled
        for j = 1:length(resp.whichAngles)
            sel(j) = cellfind(dlg(1).list, resp.whichAngles{j});
        end
        if ~isempty(resp.order), order = str2num(resp.order);
        else,                    order = 1:length(sel);
        end
        angles = angles(sel(order));
    end
end

% allow angles to be index vector into saved angles
if isnumeric(angles)
    ind = angles; % will load over the 'angles' variable below...
    angleFile = fullfile(fileparts(msh.path), 'MeshAngles.mat')
    load(angleFile, 'angles');
    angles = angles(ind);
end
  
%get the screenshots
for i = 1:length(angles)
    mrmSet(msh, 'camerarotation', angles(i).cRot);
%     images(:,:,:,i) = mrmGet(msh, 'screenshot');
    images{i} = mrmGet(msh, 'screenshot') ./ 255;
    pause(1); %empirically-needed wait, or screenshots get corrupted
end

% make the montage image
img = imageMontage(images);

% if specified, display img in a figure and add View's cbar
if cbarFlag
    hfig = figure('Color', 'w');
    imshow(img);
    
    % find the mrVista 1.0 view for the cbar
    gray = getSelectedGray;
    if isempty(gray)
        myWarnDlg('No colorbar to attach to image.')
    else
        h = gray.ui.colorbarHandle;
        set(gca, 'Position', [0 .3 1 .8]);
        hcbar = subplot('Position', [.2 .04 .6 .12]);
        tmpH = findobj('Type', 'Image', 'Parent', h);
        cbarImg = get(tmpH, 'CData');
        x = get(tmpH, 'XData'); y = get(tmpH, 'YData');
        
        himg = imagesc([x(1) x(end)], [y(1) y(end)], cbarImg);         
        
        set(hcbar, 'Box', 'off', 'Visible', get(h, 'Visible'), ...
            'XTick', get(h, 'XTick'), 'YTick', get(h, 'YTick'), ...
            'XTickLabel', get(h, 'XTickLabel'), ...
            'YTickLabel', get(h, 'YTickLabel'), ...
            'DataAspectRatio', get(h, 'DataAspectRatio'), ...
            'DataAspectRatioMode', get(h, 'DataAspectRatioMode'), ...
            'PlotBoxAspectRatio', get(h, 'PlotBoxAspectRatio'), ...
            'PlotBoxAspectRatioMode', get(h, 'PlotBoxAspectRatioMode'));
        ttl = findobj('Parent', h, 'Type', 'text', 'Rotation', 0);
        if ~isempty(ttl) % a title or xlabel exists, reproduce it
            text(0, -10, get(ttl(1), 'String'), 'FontSize', 12);
        end
        mode = sprintf('%sMode', gray.ui.displayMode);
        nG = gray.ui.(mode).numGrays;       
        colormap(gray.ui.(mode).cmap(nG+1:end,:));
    end
end

% save / export if a path is specified
if ~notDefined('savePath')
    savePath = fullpath(savePath);
    [p f ext] = fileparts(savePath);
    if isequal(lower(ext),'.ppt')
        % export to a powerpoint file
        fig = figure; imshow(img);
        [ppt, op] = pptOpen(savePath);
        pptPaste(op,fig,'meta');
        pptClose(op,ppt,savePath);
        close(fig);
        fprintf('Pasted image in %s.\n', fname);
    else
        % export to a .png image in a directory
        if isempty(f), f=sprintf('mrMesh-%s',datestr(clock)); end
        if isempty(ext), ext = '.png'; end
        fname = fullfile(p,[f ext]);
        if cbarFlag
            % export the figure w/ the cbar included
            exportfig(hfig, fname, 'Format','png', 'Color','cmyk', ...
                      'width',3.5, 'Resolution',450);
        else
            % write directly to the image
            imwrite(img, fname, 'png');
        end
        fprintf('Saved montage as %s.\n', fname);
    end
else % save to pwd
        % export to a pwd-mrMesh-date.png image in current directory
        pwdname=pwd;ll=length(pwdname)
        f=sprintf('%s-mrMesh-%s',pwdname(ll-4:ll),datestr(now,1));ext = '.png';
        fname = [f ext]
        udata.rgb = img;
        imwrite(udata.rgb, fname);
        fprintf('Saved montage as %s.\n', fname);
end

return

