
function drawROIsPerimMontage(view, lineWidth)
%
% drawROIsPerimMontage(view, [lineWidth])
% 
% Draws a line around the ROIs, for montage view.  
% This function should be called only if
% view.showROIs is non-zero.  If showROIs=-1, draw only the
% selected ROI.  If abs(showROIs)=2, draw all the ROIs.
% ras 09/04, off drawROIsPerimeter.

if ~exist('lineWidth', 'var'), lineWidth = 0.5;  end
%%%%% get info about the montage size, slice size
ui = viewGet(view,'ui');
viewType = viewGet(view,'viewType');
switch viewType
    case 'Inplane',
        firstSlice = getCurSlice(view);
		nSlices = get(ui.montageSize.sliderHandle,'Value');
        selectedSlices = firstSlice:firstSlice+nSlices-1;
        selectedSlices = selectedSlices(selectedSlices <= numSlices(view));
    case 'Flat',
        selectedSlices = getFlatLevelSlices(view);
        nSlices = length(selectedSlices);
    otherwise,
        error('drawROIsMontage: no support for this view type.');
end
nrows = ceil(sqrt(nSlices));
ncols = ceil(nSlices/nrows);
colSz = round(ui.zoom(2)-ui.zoom(1)+1);
rowSz = round(ui.zoom(4)-ui.zoom(3)+1);
xcorner = ui.zoom(3) - 1; % location of upper-right corner w/ zoom
ycorner = ui.zoom(1) - 1;

for r=1:length(view.ROIs)
    % Selected ROI: set color=white
    if (r==view.selectedROI)
        coords = view.ROIs(r).coords;
        color = viewGet(view,'selRoiColor'); 
        % Non-selected ROI, get coords, get color
    else
        % If showROIs~=2, then set coords=[] so that nothing will be
        % drawn. 
        if (view.ui.showROIs==-2)
            coords = view.ROIs(r).coords;
        else
            coords = [];
        end
        
        %color=view.ROIs(r).color;
        % We can now have many different colors in our ROIs. So we need
        % some way of translating the (more than 1-character) 'color' into
        % a color vector.
        thisCol=view.ROIs(r).color;
        if (ischar(thisCol))
            nColEntries=length(thisCol);
            
            for thisColEntry=1:nColEntries
                thisColValue=thisCol(thisColEntry);
                colRGB=[0 0 0];
                switch (thisColValue)
                    case 'y', colorRGB = colRGB+[1 1 0];
                    case 'm', colorRGB = colRGB+ [1 0 1];
                    case 'c', colorRGB = colRGB+ [0 1 1];
                    case 'r', colorRGB = colRGB+ [1 0 0];
                    case 'g', colorRGB = colRGB+ [0 1 0];
                    case 'b', colorRGB = colRGB+ [0 0 1];
                    case 'w', colorRGB = colRGB+ [1 1 1];
                    case 'k', colorRGB = colRGB+ [0 0 0];                
                    otherwise, colorRGB = [1 1 1];
                end % end switch statement
            end % end loop
            color=colorRGB./nColEntries;
        else
            color=thisCol;
        end
    end
    
    %%%%% big subloop: go through selected slices in montage
    for iSlice = 1:length(selectedSlices)
        curSlice = selectedSlices(iSlice);
        
        % Get subset of coords corresponding to the current slice
        if ~isempty(coords)
            coords = canOri2CurOri(view,coords);
            indices = find(coords(3,:)==curSlice);
            subCoords = coords(:,indices);
        else
            subCoords = [];
        end

        % Draw the lines around each pixel's edge that doesn't have a neighbor
        % w=1/2 because you want to draw the outside boarder, not the pixel
        % centers
        if ~isempty(subCoords)
            
            % The FLAT view can have a 'rotateImageDegrees' field that
            % specifies a rotation angle for each slice (L or R). 
            % If this is set, then we have to transform the ROIs by this amount as
            % well to make them register with the anatomy and functional data
            if isfield(view,'rotateImageDegrees')
                subCoords = rotateCoords(view,subCoords,0);
            end
            
            % further restrict to coordinates contained within the 
            % zoom settings:
            if checkfields(view, 'ui', 'zoom')
                Y = view.ui.zoom(3:4); % ymin ymax
                X = view.ui.zoom(1:2); % xmin xmax
                ok = find(subCoords(1,:)>=Y(1) & ...
                          subCoords(1,:)<=Y(2) & ...
                          subCoords(2,:)>=X(1) & ...
                          subCoords(2,:)<=X(2));
                subCoords = subCoords(:,ok);
            end
            
            
            % What sort of perimeter shall we draw? 'filled Perimeters' get rid
            % of the blobby effect that transforming from the volume view
            % sometimes causes.
            doFilledPerimeter=0;   
            if (isfield(view.ui,'filledPerimeter'))
                if(view.ui.filledPerimeter)
                    doFilledPerimeter=1;
                end
            end
            
            if (doFilledPerimeter)
                % Get colormap, numGrays, numColors and clipMode
                modeStr=['view.ui.',view.ui.displayMode,'Mode'];
                mode = eval(modeStr);
                cmap = mode.cmap;    
                tmat=zeros(size(view.ui.image));

                % figure out offset sizes for different
                % locations in the montage
                row = ceil(iSlice/ncols);
                col = mod(iSlice-1,ncols) + 1;
                yoffset = (col-1) * colSz;
                xoffset = (row-1) * rowSz;
                
                % get the x, y coords of the pixels
                subCoords(1,:)=subCoords(1,:) + xoffset - xcorner;
                subCoords(2,:)=subCoords(2,:) + yoffset - ycorner;    

                % clip subCoords: may not be necessary for montage
                subCoords=round(subCoords(1:2,:));
                [msize]=max(size(view.ui.image));
                subCoords(subCoords<1)=1;
                subCoords(subCoords>msize)=msize;
                
                % These are the indices into the image of the ROI
                inds=sub2ind((size(view.ui.image)),subCoords(1,:),subCoords(2,:));
                tmat(inds)=1;
                
                % Use the image processing toolbox to dilate, erode, fill the
                % ROI.
                se=strel('disk',4,4);
                tmat=imdilate(logical(tmat),se);
                se=strel('disk',4,4);
                tmat=imerode(logical(tmat),se);
                tmat=imfill(tmat,'holes');
                tmat=tmat-min(tmat(:));
                tmat=bwperim(tmat);
                onpoints=find(tmat);
                
                [y x]=ind2sub((size(view.ui.image)),onpoints);
                
                %line(b(:,1),b(:,2),'Color',color,'LineWidth',lineWidth);
                hold on;
                lineVect=plot(x,y,'.','LineWidth',lineWidth*2);
                for t=1:length(lineVect)
                    set(lineVect,'Color',color,'Marker','.','MarkerSize' ,lineWidth*15);
                end
                
                
            else % Do the regular perimeters
                % figure out offset sizes for different
                % locations in the montage
                row = ceil(iSlice/ncols);
                col = mod(iSlice-1,ncols) + 1;
                yoffset = (col-1) * colSz;
                xoffset = (row-1) * rowSz;
                
                % get the x, y coords of the pixels
                w=0.5;
                x=subCoords(1,:) + xoffset - xcorner;
                y=subCoords(2,:) + yoffset - ycorner;    
                
                hold on
                for i=1:size(subCoords,2);
                    xMinus = find(x == x(i)-1);
                    xEquals = find(x == x(i));
                    xPlus = find(x == x(i)+1);
                    if isempty(xMinus)
                        line([y(i)-w,y(i)+w],[x(i)-w, x(i)-w],'Color',color,'LineWidth',lineWidth);
                    else
                        if ~any(y(i) == y(xMinus))
                            line([y(i)-w,y(i)+w],[x(i)-w, x(i)-w],'Color',color,'LineWidth',lineWidth);
                        end
                    end
                    if isempty(xPlus)
                        line([y(i)-w,y(i)+w],[x(i)+w, x(i)+w],'Color',color,'LineWidth',lineWidth);
                    else
                        if ~any(y(i) == y(xPlus))
                            line([y(i)-w,y(i)+w],[x(i)+w, x(i)+w],'Color',color,'LineWidth',lineWidth);
                        end
                    end
                    if ~isempty(xEquals)
                        if ~any(y(i) == y(xEquals)-1)
                            line([y(i)+w,y(i)+w],[x(i)-w, x(i)+w],'Color',color,'LineWidth',lineWidth);
                        end
                        
                        if ~any(find(y(i) == y(xEquals)+1))
                            line([y(i)-w,y(i)-w],[x(i)-w, x(i)+w],'Color',color,'LineWidth',lineWidth);
                        end  
                    end
                end  % drawing reg. perimeters
               
            end     % filled vs. reg perimeter if statemnt
            hold off
        end     % if ~isempty(subCoords)
    end     % loop through selected slices
end 

return;
