function [vANATOMYPATH, anatDir, subject] = getVAnatomyPath(varargin);
%
% [vANATOMYPATH, anatDir, subject] = getVAnatomyPath;
%
% Gets path of a given subject's volume anatomy. No longer takes any
% input arguments (ignores them all): just looks down a standard list
% of possible paths for a volume anatomy file, returns the first one
% it finds, and if it doesn't find any, prompts the user. 
%
% The order of paths to check is:
%   (1) variable vANATOMYPATH saved in mrSESSION file
%   (2) if a user default path is specified (via the command
%       "setpref VISTA defaultAnatomyPath [path]"), gets that 
%       path and checks for [defaultPath]/vAnatomy.dat
%   (3) local link (in HOMEDIR) 3Danatomy/vAnatomy.dat  [KGS-lab]
%   (4) local link (in HOMEDIR) anat/vAnatomy.dat  [KGS-lab]
%   (5) X:\anatomy\[mrSESSION.subject]\vAnatomy.dat or
%       /biac1/wandell/data/[Wandell lab]
%   (6) prompt user
%
% OUTPUTS:
%   vANATOMYPATH: full path to volume anatomy file.
%   anatDir: anatomy directory containing file.
%   subject: subject's name, from mrSESSION.subject field.
%
% 2.26.99 - Written by Press
% djh, 2/15/2001 - modified to use fullfile
% djh, 9/26/2001 - guess subject from path if it isnt already known
% djh, 8/17/2002 - added queryFlag
% jl,bw -- added comment about setting X: mount point.
% ras, 03/05 - saves to/retrieves from mrSESSION file.
% ras, 12/05 - LARGE CHANGE: honestly, in 5 years of using this program, 
% this function has not once successfully executed for me. KGS lab has been 
% using our own version for a while, but in an attempt to not fork, I've
% made some large-scale changes. These incorporate the Wandell-lab 
% conventions, KGS-lab conventions, and the recent addition of having
% a preference variable for the path.

% prior to checking, let's make sure we have everything loaded up we need
mrGlobals; % adds HOMEDIR, mrSESSION global variables
if ieNotDefined('HOMEDIR'), HOMEDIR = pwd; end
mrSessPath = fullfile(HOMEDIR, 'mrSESSION.mat');
if exist(mrSessPath, 'file'), load(mrSessPath); end

% (1) see if it's saved in the mrSESSION file.
if ieNotDefined('vANATOMYPATH');
    % initalize empty path
    vANATOMYPATH = '';
end

% (2) check for user-set default anatomy path 
if ~exist(vANATOMYPATH, 'file')
    if ispref('VISTA', 'defaultAnatomyPath')
        test = getpref('VISTA', 'defaultAnatomyPath');
        vANATOMYPATH = fullfile(test, 'vAnatomy.dat');
    end
end

% (3) check for 3DAnatomy/vAnatomy.dat
if ~exist(vANATOMYPATH, 'file')
    if exist(fullfile(HOMEDIR, '3DAnatomy'), 'dir')
        vANATOMYPATH = fullfile(HOMEDIR, '3DAnatomy', 'vAnatomy.dat');
    end
end

% (4) check for anat/vAnatomy.dat
if ~exist(vANATOMYPATH, 'file')
    if exist(fullfile(HOMEDIR, 'anat'), 'dir')
        vANATOMYPATH = fullfile(HOMEDIR, 'anat', 'vAnatomy.dat');
    end
end

% (5) check for X:/anatomy/[subject]/vAnatomy.dat 
%     or /biac1/wandell/[subject]/vAnatomy.dat
if ~exist(vANATOMYPATH, 'file')
    if ispc
        test = fullfile('X:\anatomy\', mrSESSION.subject);
    else
        test = fullfile('/biac1/wandell/', mrSESSION.subject);
    end
    
    vANATOMYPATH = fullfile(test, 'vAnatomy.dat');
end

% (6) still not found: prompt user
if ~exist(vANATOMYPATH, 'file')
    [f p] = uigetfile({'*.dat', '*.*'}, 'Select a Volume Anatomy File');
    vANATOMYPATH = fullfile(p, f);
end

% *** Save the specified path in mrSESSION ***
save(mrSessPath, 'vANATOMYPATH', '-append');

% Also set as a global variable
evalin('base', 'global vANATOMYPATH;');
assignin('base', 'vANATOMYPATH', vANATOMYPATH);

anatDir = fileparts(vANATOMYPATH);
subject = mrSESSION.subject;

return;