function [vSig, pSig, ces] = er_stxgrinder(Test,hAvg, voxVar, resVar, DOF, RM, q)
%
% [vSig, <pSig>, <ces> ] = er_stxgrinder(Test, hAvg, voxVar, resVar, DOF,
% RM, q)
%
% Inputs:
%   Test: char, either 'T' or 'F', determining what type of test to run run
%
%   hAvg: Average hemodynamic responses, contained within the results of
%   er_selxavg (or betas from glm).
%
%   voxVar: sum over scans of Xt * inv(C) * X, where X is the 
%   design matrix and C is the "whitening" matrix (identity
%   matrix of size nTimePoints if not whitening). Sometimes
%   called the 'hemodynamic covariance matrix'.
%
%   resVar: residual variance estimate, computed as the sum-square
%   of the model's residuals (model.residual in glm) divided
%   by the degrees of freedom
%
%   DOF: degrees of freedom
%
%   RM: restriction matrix -- see glm_restrictionMatrix
%
%   q: some sort of idealized value which gets subtracted from the rows of
%   hAvg. [See Burock and Dale, HBM, 2000]
%   
% Outputs:
%   vSig: Value of the requested statistic (T or F, depending on the
%         test).
%   pSig: p-value associated with the value vSig, the associated
%         degrees of fredom of the model, and the selected probability
%         distribution function (T or F).
%   ces:  contrast effect size, as a difference of betas weighted
%         by the restriction matrix.
%
%
% $Id: er_stxgrinder.m,v 1.8 2006/01/06 12:37:34 sayres Exp $
% 10/1/03 ras: copied from fmri_stxgrinder, currently unmodified, just a
% local version for mrLoadRet, to remove dependencies on the FS-FAST
% toolbox.
if (nargin < 6)
    msg = ...
        'USAGE: [vSig, pSig, ces] = er_stxgrinder(Test, hAvg, voxVar, resVar,DOF, RM , <q>)';
    error(msg);
end

if (size(RM,2) ~= size(hAvg,3))
    fprintf('ERROR: dimension mismatch between the contrast matrix\n');
    fprintf('and the input averages. This may have happened if \n');
    fprintf('you redefined an analysis without redefining the \n');
    fprintf('contrast. Try re-running er_mkcontrast.\n');
    return;
end

%%%% -------  Determine the Test Type ------------ %%%%%%
if (strncmp('T',upper(Test),1))     TestId = 0; % t-Test
elseif (strncmp('F',upper(Test),1)) TestId = 1; % F-Test
else  error(sprintf('Test Type %s is unknown',Test));
end

%%% -- get the dimensions --- %%%
nRows = size(hAvg,1);
nCols = size(hAvg,2);
nT    = size(hAvg,3);
nV = nRows * nCols;

%% ---- Subtract the Ideal, if needed ----- %%%%%
if ( nargin == 6 )
    h = hAvg;
else
    if (size(q,1) ~= size(hAvg,3))
        error('hAvg and q must have the same number of rows');
    end
    tmp = reshape(repmat(q, [nV 1]), [nRows nCols nT]);
    h = hAvg - tmp;
    clear tmp;
end

ind0 = find(voxVar==0);
l0 = length(ind0);
% fprintf(1,'  nVoxels with voxVar=0: %3d\n',l0);
if (l0 == prod(size(voxVar)))
    fprintf(1,'INFO: All voxels are zero\n');
    if (TestId == 0)
        sz = [nRows nCols size(RM, 1)];
    else
        sz = [nRows nCols 1];
    end
    pSig = ones(sz);
    vSig = zeros(sz);
    ces =  zeros(sz);
    return;
end
if (l0 ~= 0)
    indnz = find(voxVar ~= 0);
    VoxVar_min = min(reshape1d(voxVar(indnz)));
    voxVar(ind0) = VoxVar_min;
    fprintf(1,'  Setting zeros to min=%g\n',VoxVar_min);
end


%%%% ----- reshape for easier processing ----- %%%%%
h     = reshape(h, [nV nT])'; %'
VoxVarB = reshape(voxVar, [nV 1])'; %'

%%% ---- Compute inv of DOF/Desgin covariance mtx --- %%%%
RChRt   = RM * resVar * RM'; %'
ind = find(RChRt==0);
RChRt(ind) = .0000000000001;

% Compute contrast effect size %
ces = RM*h;
nces = size(ces,1);
ces = reshape(ces',[nRows nCols nces]);

%%% --- Perform Tests --- %%%
if (TestId == 0) 
    % t-Test
    dRChRt = diag(RChRt);
    vSig = (RM * h) ./ sqrt(dRChRt * VoxVarB);
    
    pSig = er_ttest(DOF, vSig(:), 300); % dof>300 --> normal approx %
    pSig = reshape(pSig, size(vSig));
    
else  
    % F-Test
    if (strcmp('FM',upper(Test))) 
        dRChRt = diag(RChRt);
        vSig = (RM * h).^2 ./ (dRChRt * VoxVarB);
        J = 1;
    else
        RvvR = RM' * inv(RChRt) * RM; %'
        [J Nh] =  size(RM); % Rows of RM %
        if (Nh==1)
            vSig =  (((h' * RvvR)' .*h) ./ VoxVarB)/J;
        else
            vSig =  ((sum((h' * RvvR)' .* h)) ./ VoxVarB)/J;
        end
    end
    
    pSig = er_ftest(J, DOF, reshape1d(vSig), 1000); % 1500 = maxdof2
    pSig = reshape(pSig, size(vSig));
end

%% Reshape into image dimensions %%
vSig = reshape(vSig', [nRows nCols size(vSig,1)]); %'
pSig = reshape(pSig', [nRows nCols size(pSig,1)]); %'


return;
