function stim = er_concatParfiles(view,scans);
% stim = er_concatParfiles(view,[scans]);
%
% Reads in the parfiles assigned to the selected scans
% in view, concatenates them together, and returns the
% results in an stim struct.
%
% Fields of the stim struct:
%   'onsetSecs': onsets for each trial in seconds.
%   'cond': condition # of each trial
%   'label': text label, if any, specified for each trial
%   'run': run from which each trial was taken
%   'TR': the frame period of the scans
%   'parfiles': names of the parfiles from which the info was taken
%   'onsetFrames': onsets for each trial in fMRI frames.
%   'condNums': the unique #s used to specify different conditions in
%               the selected parfiles.
%   'condNames': The names for each condition, taken from the first
%                trial of each #
%
%
%
% 04/27/04 ras.
% 09/04 ras: made paths relative to HOMEDIR, so you don't have
% to be cd'd into it when you call this.
global dataTYPES HOMEDIR;

if ieNotDefined('view')
    view = getSelectedInplane;
end

if ieNotDefined('scans')
    [scans dt] = er_getScanGroup(view);
    view = viewSet(view,'curDataType',dt);
end

dt = view.curDataType;
TR = dataTYPES(dt).scanParams(scans(1)).framePeriod;

stim.onsetSecs = [];
stim.cond = [];
stim.label = [];
stim.color = [];
stim.run = [];
stim.TR = TR;

for s = 1:length(scans)
    parList{s} = dataTYPES(dt).scanParams(scans(s)).parfile;
    if ~exist(parList{s}, 'file')
        % try appending .par
        [p f ext] = fileparts(parList{s});
        if isempty(ext), parList{s} = fullfile(p, [f '.par']); end
    end
    
    if ~exist(parList{s}, 'file')
        % probably relative to parfilesDir, try this:
        parList{s} = fullfile(parfilesDir(view), parList{s});
    end
    
    if ~exist(parList{s}, 'file')
        error(sprintf('Couldn''t find file %s. ', parList{s}));
    end
end

stim.parfiles = parList;

for s = 1:length(scans)
    [onsets, conds, labels, colors] = readParFile(parList{s});
    
    % ignore all entries w/ negative time onsets, or negative cond #s
    stim.framesPerRun(s) = dataTYPES(dt).scanParams(scans(s)).nFrames;
    maxt = stim.framesPerRun(s) * TR;
    ok = find(conds>=0 & onsets>=0 & onsets<maxt);
    onsets = onsets(ok);
    conds = conds(ok);
    labels = labels(ok);
    colors = colors(ok);  
    
    % add a final entry specifying the last frame as cond 0
    conds(end+1) = 0;
    labels{end+1} = 'end of run';
    onsets(end+1) = (stim.framesPerRun(s)-1) * TR;
    if ~isempty(cellfind(colors)) 
        colors{end+1} = [0 0 0];
    end

    % increment onsets for scans after the first
    if s > 1
        framesInLastScan = dataTYPES(dt).scanParams(scans(s-1)).nFrames;
        onsets = onsets + (s-1)*TR*framesInLastScan; % stim.onsetFrames(end);
    end
    
    stim.onsetSecs = [stim.onsetSecs onsets];
    stim.cond = [stim.cond conds];
    stim.label = [stim.label labels];
    stim.color = [stim.color colors];
    stim.run = [stim.run s*ones(size(onsets))];
end

% convert to frame #s
stim.onsetFrames = round(stim.onsetSecs ./ TR); 
stim.onsetFrames = stim.onsetFrames + 1;

% also return the names of each unique
% condition # used, and the corresponding 
% labels for each condition
stim.condNums = unique(stim.cond(stim.cond>=0));
nConds = length(stim.condNums);
for i = 1:nConds
    ind = find(stim.cond==stim.condNums(i));
    stim.condNames{i} = stim.label{ind(1)};
    stim.condNames{i}(stim.condNames{i}=='_') = ' ';
end
stim.condNames = deblank(stim.condNames);

% for any blank condition names, assign a string with the number
unassigned = cellfind(stim.condNames, '');
for i = unassigned, stim.condNames{i} = num2str(stim.condNums(i)); end

% get condition colors:
% now, I've implemented two different ways to save this info. 
% one saves it in the event-related params, the other in the parfiles.
% I'd like to phase out the former way (from params), so use
% it only if it's set in params but not in the parfiles
params = er_getParams(view,scans(1));
if ~isempty(cellfind(colors))
    for i = 1:nConds
        ind = find(stim.cond==stim.condNums(i));
        stim.condColors{i} = stim.color{ind(1)};
    end
elseif isfield(params,'condColors') & ~isempty(params.condColors)
    stim.condColors = params.condColors;
else
    stim.condColors = tc_colorOrder(length(stim.condNums));
end

% one last pass: check for leftover conds w/ unassigned colors
defaults = tc_colorOrder(nConds);
leftover = setdiff(1:nConds, cellfind(stim.condColors));
for i=leftover, stim.condColors{i} = defaults{i}; end


return
