function tc = tc_visualizeGlm(tc, parent);
%
% tc = tc_visualizeGlm(tc, parent);
%
% Provide graphics that visualize the
% setup and result of applying a general
% linear model to a time course. Requires
% that you first run tc_applyGlm on
% the tc time course struct.
%
%
% ras 04/05.
if nargin<1,    tc = get(gcf,'UserData');     end
if nargin<2,    parent = tc.ui.plot;          end
if parent==gcf | parent==get(gcf, 'CurrentAxes')
    % make a uipanel to fit on the target
    parent = uipanel('Parent', parent, ...
        'Units', 'normalized', ...
        'BackgroundColor', get(gcf, 'Color'), ...
        'Position', [0 0 1 1])
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% apply a GLM if needed
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isfield(tc, 'glm')
    tc = tc_applyGlm(tc);
end
X = tc.glm.designMatrix;
Y = tc.wholeTc(:);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% clean up existing objects in figure
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
otherAxes = findobj('Type', 'axes','Parent', parent);
delete(otherAxes);
otherUiControls = findobj('Type', 'uicontrol', 'Parent',parent);
delete(otherUiControls);
axes('Parent', parent); % shift focus to parent uipanel
delete(gca);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Visualize results: diff't options for deconvolved
% data and non-deconvolved (using HRF)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if isequal(tc.glm.type, 'selective averaging')
    tc = tc_plotDeconvolvedTCs(tc, parent);
    return
end

% if we got here, the GLM is non-deconvolved, using HRF
% show hemodynamic response function used for GLM
axes('Parent', parent, 'Position', [.13 .58 .21 .34]);
plot(tc.glm.hrf,'k','LineWidth',2);
grid on
xlabel('Time, frames')
ylabel('Arbitrary Units')
if isnumeric(tc.params.glmHRF)
    opts = {sprintf('Mean trial for conditions %s',num2str(tc.params.snrConds)), ...
        'Boynton gamma function', 'SPM difference-of-gammas' ...
        'Dale & Buckner ''97'};
    hrfName = opts{tc.params.glmHRF};
else
    hrfName = tc.params.glmHRF; hrfName(hrfName=='_') = ' ';
end
title(sprintf('HRF function used: %s', hrfName))


% show design matrix
axes('Parent', parent, 'Position', [.13 .11 .21 .34]);
Pred = X .* 2 .* repmat(tc.glm.betas,[size(X,1) 1]);
imagesc(X);
colormap gray
nConds = sum(tc.trials.condNums>0);
tickPts = [1:nConds];
tickLabels = {'Individual Conditions' 'DC Predictors for each run'};
set(gca,'XTick',tickPts);
xlabel('Predictors (Conditions + DC)')
ylabel('Time, Frames')
title('Design Matrix')
% colorbar vert

% bar beta values for selected conditions
axes('Parent', parent, 'Position', [.57 .58 .33 .34]);
sel = find(tc_selectedConds(tc));
tmp = tc.trials.condColors(sel);
for i = 1:length(sel), col(i,:) = tmp{i}; end
mybar(tc.glm.betas(sel-1), tc.glm.sems(sel-1),...
    tc.trials.condNames(sel), [], col);
% xlabel Predictors
ylabel('% Signal Change')
title('Beta Values');

% show time course + selected predictors
axes('Parent', parent, 'Position', [.57 .11 .33 .34]);

% Pred2 will contain scaled predictors, plus the
% baseline value during each scan, to look nicer
Pred2 = Pred(:,1:nConds);
for i = nConds+1:size(X,2)
    ind = find(X(:,i)>0);
    Pred2(ind,:) = Pred2(ind,:) + mean(Pred(ind,i));
end
for i = 1:nConds
    % normalize in case of HRF undershoot, by the 1st frame
    % (Gum...)
    Pred2(:,i) = Pred2(:,i) - Pred(1,i);
end

t = [1:length(tc.wholeTc)] .* tc.TR;
hold on
plot(t,tc.wholeTc(:),'k-.','LineWidth',1.5);
plot(t,Pred2);
setLineColors([{'k'} tc.trials.condColors(sel)]);
xlabel('Time, sec')
ylabel('% Signal')
title('Time course + Scaled Predictors')
zoom

% for longer time courses, may want a UI control
scale = 200;  % max seconds to nicely plot TC
if max(t) > scale
    % shrink the axes a little for the control
    oldPos = get(gca,'Position');
    newPos = [oldPos(1) oldPos(2)+.08 oldPos(3) oldPos(4)-.08];
    set(gca,'Position',newPos);

    % set the axes to have a nice aspect ratio
    AX = axis;
    newAX = [1 scale AX(3) AX(4)];
    axis(newAX);

    % add a slider to move back and forth
    cbstr = 'val=get(gcbo,''Value'');';
    cbstr = sprintf('%s \n axis([val val+%i %f %f]);',...
        cbstr,scale,AX(3),AX(4));
    
    sliderPos = [oldPos(1) .02 oldPos(3) .05];
    if tc.params.legend, sliderPos = sliderPos + [-.15 0 -.05 0]; end
    h1 = uicontrol('Style','slider', 'UserData', gca, ...
        'Units','Normalized', 'Position', sliderPos, ...
        'Min', 0, 'Max', max(t)-scale, ...
        'Callback',cbstr,'BackgroundColor','w');
    
    zoomPos = [oldPos(1)-.1 .02 .08 .05];
    if tc.params.legend, zoomPos = zoomPos + [-.12 0 -.02 0]; end    
    h2 = uicontrol('Style','pushbutton', 'Units', 'Normalized',...
        'Position', zoomPos, ...
        'BackgroundColor', 'w', 'String', 'Whole Tc', ...
        'Callback','axis auto');

end

return
% /-------------------------------------------------------------------/ %




% /-------------------------------------------------------------------/ %
function tc = tc_plotDeconvolvedTCs(tc, parent);
% plot the deconvolved time courses in the same way as for the
% default plot, combining mean amplitudes on one side and mean time
% courses on the other.
sel = find(tc_selectedConds(tc));
sel = sel(sel>1); % no baseline estimated
nConds = length(sel);
frameWindow = unique(round(tc.timeWindow./tc.TR));
prestim = -1 * frameWindow(1);
peakFrames = unique(round(tc.peakPeriod./tc.TR));
bslFrames = unique(round(tc.bslPeriod./tc.TR));
peakFrames = find(ismember(frameWindow,peakFrames));
bslFrames = find(ismember(frameWindow,bslFrames));

%%%%%%%%%%%%%%%%%%%%%%%%
% calc mean amplitudes %
%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% norm baseline periods
deconvolvedTcs = tc.glm.betas(:,sel-1);
deconvolvedSems = tc.glm.sems(:,sel-1);
for i = 1:nConds
    offset = mean(deconvolvedTcs(bslFrames, i));
    deconvolvedTcs(bslFrames,i) = deconvolvedTcs(bslFrames,i) - offset;
end

%%%%% calc amplitudes
for i = 1:nConds
    bsl = deconvolvedTcs(bslFrames, i);
    peak = deconvolvedTcs(peakFrames, i);
    amps(:,i+1) = (nanmean(peak) - nanmean(bsl))';

    sems(:,i+1) = nanmean(tc.glm.sems(:,i))';
end

%%%%%%%%%%%%%%%%%%%%%%%%
% plot mean amplitudes %
%%%%%%%%%%%%%%%%%%%%%%%%
axes('Position', [.1 .2 .35 .6], 'Parent', parent);
hold on

h1 = gca;

lineWidth = 2;
labels = {};

colors = tc.trials.condColors(sel);
for x = 1:nConds
    starbar(amps(x), sems(x), [], 'X', x, 'color', colors{x});
end
grid on

% set line width
htmp = findobj('Type','line','Parent',gca);
set(htmp,'LineWidth',lineWidth);

% add labels
set(gca,'XTick',[1:nConds]);
if tc.params.legend==0, set(gca, 'XTickLabel', labels); end
xlabel('Condition', 'FontWeight', 'bold', 'FontAngle', 'italic');
ylabel('Mean Amplitude, % Signal', 'FontWeight', 'bold', ...
    'FontAngle', 'italic');
title('Deconvolved Amplitudes', 'FontWeight', 'bold');

% set axes to frame bars nicely
% axis auto;
AX = axis;
AX(1:2) = [0 nConds+1];
if isfield(tc.params,'axisBounds') & ~isempty(tc.params.axisBounds)
    AX(3:4) = tc.params.axisBounds(3:4);
end
axis(AX);

%%%%%%%%%%%%%%%%%%%%%
% mean time courses %
%%%%%%%%%%%%%%%%%%%%%
h2 = axes('Position', [.5 .2 .4 .6], 'Parent', parent);
hold on

for i = 1:nConds
    cond = tc.trials.condNums(sel(i));
    col = tc.trials.condColors{sel(i)};
    htmp = errorbar(tc.timeWindow, deconvolvedTcs(:,i), deconvolvedSems(:,i));
    set(htmp, 'Color', col, 'LineWidth', 2);
end

%grid on

% indicate the peak and baseline periods, if selected
if tc.params.showPkBsl==1
    AX = axis;
    plot(tc.bslPeriod, repmat(AX(3),size(tc.bslPeriod)), ...
        'k', 'LineWidth', 3.5);
    plot(tc.peakPeriod, repmat(AX(4),size(tc.peakPeriod)), ...
        'r', 'LineWidth', 3.5);
end
grid on

if isfield(tc.params,'axisBounds') & ~isempty(tc.params.axisBounds)
    axis(h2, tc.params.axisBounds);
end

xlabel('Trial time, secs', 'FontWeight', 'bold', 'FontAngle', 'italic');
ylabel('% Signal', 'FontWeight', 'bold', 'FontAngle', 'italic');
title('Deconvolved Time Courses', 'FontWeight', 'bold');

return


