function tc = tc_init(view, roi, scans, dt);
% tc = tc_init(view, roi, scans, dt);
%
% Initialize a time course UI struct from a mrVista view.
%
% roi: name or # of ROI in the view.ROIs substruct. 
%
% ras, broken off from tc_openFig, 04/06/05.
global dataTYPES mrSESSION;
if notDefined('view'),   view = getSelectedInplane;           end
if notDefined('roi'),    roi = viewGet(view,'curRoi');        end
if notDefined('dt'),     dt = viewGet(view,'curDataType');    end
if notDefined('scans')
    [scans dt] = er_getScanGroup(view);
    view = selectDataType(view,dt);
end

% get prefs
if ispref('VISTA','recomputeVoxData') 
    recomputeFlag = getpref('VISTA','recomputeVoxData');
else
    recomputeFlag = 0;
end

tic

% get roi substruct 
roi = tc_roiStruct(view,roi);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% check if data are from a different data type
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ischar(dt)
    % convert dt name -> index #
    mrGlobals;
    dt = existDataType(dt,dataTYPES,1);
end

if ~isequal(dt,viewGet(view,'curDataType'))
    names = {dataTYPES.name};
    msg = sprintf('The selected scans are: \n %s %s\n',...
            names{dt},num2str(scans));
    msg = [msg 'This is in a different data type. '];
    msg = [msg 'Do you want to analyze this data? '];
    resp = questdlg(msg,'Time Course UI','Yes','No','No');
    if ~isequal(resp,'Yes')
        error('TC UI init aborted.')
    end
    
    % so as not to affect where the GUI is pointed, reset 
    % the 'view' argument as a hidden view, of the appropriate
    % type, set to the right data type and scan:
    switch view.viewType
        case 'Inplane', view = initHiddenInplane(dt, scans(1));
        case 'Volume', view = initHiddenVolume(dt, scans(1));
        case 'Gray', view = initHiddenGray(dt, scans(1));
        case 'Flat', view = initHiddenFlat(dt, scans(1));
    end
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% check if parfiles are assigned (if ABAB block design or cyclic, make 'em)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
status = tc_parfilesCheck(view, scans);
if status==0,
    tc = []; disp('No parfiles assigned -- can''t initialize tc');
    return
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% load the data from scans, parfiles
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% tc = er_chopTSeries(view,roi.coords,scans,'waitbar');

% alternate way: use voxelData function, then avg 
% across voxels.
% b/c this saves/loads ROI tSeries, it may be much
% faster in many circumstances
data = er_voxelData(view, roi, scans, recomputeFlag);
tc.wholeTc = nanmeanDims(data.tSeries, 2);
tc = er_chopTSeries2(tc.wholeTc, data.trials, data.params);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% initialize additional fields in tc struct
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
tc.TR = dataTYPES(dt).scanParams(scans(1)).framePeriod;
tc.plotType = 7; % default: mean amps plus tcs
tc.condNames = {};
tc.condColors = {};
tc.params = er_getParams(view,scans(1));
tc.params.parfile = [];
tc.params.scans = scans;
tc.params.dataType = dataTYPES(dt).name;
tc.params.viewName = view.name;
tc.params.sessionCode = mrSESSION.sessionCode;
tc.params.legend = 1;
tc.params.showPkBsl = 1;
tc.params.deconvolved = 0;
tc.params.parfiles = {dataTYPES(dt).scanParams(scans).parfile};
tc.trials = er_concatParfiles(view,scans);
tc.condNames = tc.trials.condNames; % back-compatibility
tc.condNums = tc.trials.condNums; % back-compatibility
% tc = tc_parseParfile(tc,view,scans); % adds onset,conds
nConds = length(tc.condNums);
colors = tc_colorOrder(nConds);
for i = 1:nConds
    j = mod(i-1,length(colors))+1;
    tc.condColors{i} = colors{j};
end

% check if we're in a deconvolved data type
curdt = viewGet(view,'curDataType');
dtName = dataTYPES(curdt).name;
if isequal(dtName,'Deconvolved')
    % fix to be compatible with deconvolved format
    tc = tc_deconvolvedTcs(view,tc);
end

tc.roi = roi;

toc

return

