function [X, nh, hrf] = glm_createDesMtx(stim, params, tSeries, reshapeFlag);
%
% [X, nh, hrf] = glm_createDesMtx(stim, [params], [tSeries or nFrames], [reshapeFlag]);
%
% Create a design matrix appropriate for the hrf option
% specified in the selected event-related params 
% (see er_getParams for more info on the params struct).
%
% If the params.glmHRF option is 0, the design matrix X will
% be appropriate for a deconvolution GLM in which time 
% courses relative to trial onset for each condition are 
% estimated as part of the GLM. If this flag is positive (1-3),
% a design matrix X will be returned in which there is a single
% predictor function for each condition, using an assumed form
% of the hemodynamic response function (HRF).
%
% Entering the optional tSeries argument is needed if the
% params.glmHRF option is 1 -- estimate HRF from mean 
% response to all stim. If entered, the design matrix
% will also be clipped to the # of frames in the tSeries.
% You can enter the # of frames directly as the third 
% argument instead of the whole tSeries if you are using
% a different HRF option, but want to clip to the # of frames.
%
% The optional reshape flag, if set to 1 [default 0], will 
% cause the otherwise-2D matrix to be set as a 3D matrix
% with the third dimension being different scans. This is 
% appropriate for much of the new GLM code.
%
% Also returns nh, the # of time points to use in the 
% hemodynamic response window for estimating a GLM.
% (see glm, applyGlm); and hrf, the response function
% used (empty if deconvolving).
%
%
% ras, 04/05
% ras, 03/06: divides each predictor column by the max absolute value
% of that column, so they range between -1 and 1 (but 0 values are 
% unchanged). We may want to set a parameter for this (in which case, it should
% logically go into params, like 'params.glmNormalize', but I'm 
% confident enough that this is the right move that I'm going to 
% not make it do this). -ras, 03/01/01
if notDefined('params'),      params = er_defaultParams;  end
if notDefined('reshapeFlag'), reshapeFlag = 0;            end
if notDefined('tSeries'),     tSeries = [];               end

tr = params.framePeriod;

if ~isfield(params, 'rmTrend'), params.rmTrend = 1;       end

% figure out whether an entire tSeries
% was passed, or just a # of frames:
if length(tSeries)==1
    % nFrames is specified, rather than tSeries
    nFrames = tSeries;
    tSeries = [];
elseif ~isempty(tSeries)
   nFrames = size(tSeries,1);
   if nFrames==1
       % need it as a column vector
       tSeries = tSeries';
       nFrames = size(tSeries,1);
   end
end

if isempty(tSeries)
	% default is max frames specified in stim struct
    nFrames = stim.onsetFrames(end);
end

% we allow the params.glmHRF to specify the filename of 
% a saved HRF (will use the hrfDir function to find its path)
if ischar(params.glmHRF)
    [p f ext] = fileparts(params.glmHRF);
    hrfPath = fullfile(hrfDir, [params.glmHRF '.mat']);
    if ~exist(hrfPath, 'file')
        error(sprintf('Couldn''t find HRF file %s.', hrfPath));
    end
    
    % for now, we set a flag for the switch below: we're applying
    % this HRF
   params.glmHRF = 5;
end

% decide whether we're deconvolving (getting estimated time courses 
% for each condition) or fitting an HIRF (getting only a single beta value
% for each condition) based on the selected event-related hrf parameter. 
% Get a corresponding stimulus matrix:
if params.glmHRF==0
    %%%%%%%%%%%%%%%%
    % deconvolving %
    %%%%%%%%%%%%%%%%
    framesPerScan = max(stim.framesPerRun);
    
    % make a delta-function matrix for onsets of 
    % different conditions:
    s = delta_function_from_parfile(stim.parfiles, tr, framesPerScan);
    
    % create Toeplitz matrix for estimating deconvolved responses
    % (see papers on 'selective-averaging', e.g. by Randy Buckner
    % or Douglas Greve for the Freesurfer code)
    fw = unique(round(params.timeWindow/tr)); % time window
    hrf = [];
    nConds =  size(s,2); 
    [X nh] = glm_deconvolution_matrix(s, fw);
    
    % also add a 'trend-removal' component, to mirror the trend removal
    % in the FS-FAST processing stream (see fast_trendmtx):
    if params.rmTrend==1
        nScans = length(stim.parfiles);
        Xtrend = zeros(framesPerScan, nScans, nScans);
        for i = 1:nScans
            v = [0:framesPerScan-1]'; 
            v = v - mean(v);
            v = v./sqrt(sum(v.^2));
            Xtrend(:,i,i) = v;
        end
        X = cat(2, X, Xtrend);
    end

    % reshape to 2D (will undo later if needed)
    X = reshape(permute(X,[1 3 2]),[size(X,1)*size(X,3) size(X,2)]);
else
    %%%%%%%%%%%%%%%%
    % applying HRF %
    %%%%%%%%%%%%%%%%
    % first, construct the impulse response function: 
    switch params.glmHRF
        case 1, 
            % estimate hrf as mean time course to all stim
            if isempty(tSeries)
                errmsg = 'Need tSeries to estimate mean trial response (glmHRF option 1)';
                error(errmsg);
            end
            
            disp('Estimating mean trial response for GLM')
            tSeries = squeeze(mean(tSeries,2)); % mean across voxels
            anal = er_chopTSeries2(tSeries(:),stim,params);
            postStim = find(anal.timeWindow>=0);
            hrf = anal.meanTcs(postStim,anal.condNums>0);
            hrf = hrf(:,params.snrConds);
            if size(hrf,2) > 1
                hrf = nanmeanDims(hrf,2);
            end
            hrf = hrf ./ sum(hrf); % normalize to integral 1
        case 2,  % boynton gamma function
            hrf = 'boynton';
        case 3,  % spm difference-of-gammas
            hrf = 'spm';
        case 4, % dale & buckner HRF
            hrf = 'dale';
        case 5, % saved HRF from file
            load(hrfPath, 'hrf', 'timeWindow', 'tr');            
            % TO DO: deal w/ situations in which the HRF was
            % saved using diff't TR, time window

        otherwise, error('Invalid specification for params.glmHRF.')
    end
    
    % apply HRF -- return a 2D matrix covering whole time course
    [X, hrf] = glm_convolve(stim, params, hrf, nFrames);            
    
    % we're only returning one predictor per condition: the
    % nh variable should note this:
    nh = 1;
end


% % Norm axes of each column, such that the max positive value is 1, 
% % the max negative value is -1, but 0 values remain 0. 
% % This will ensure that beta values reflect the units of the input
% % data (time series).
% for i = 1:size(X, 2)
%     X(:,i) = X(:,i) ./ max(abs(X(:,i)));
% end


if reshapeFlag==1
    % return a 3D matrix w/ runs as the 3rd dimension
    
    % figure out max # frames per run to 
    % use for reshaping
    for run = unique(stim.run)
        ind = find(stim.run==run);
        onsets{run} = stim.onsetFrames(ind);
        framesPerRun(run) = onsets{run}(end)-onsets{run}(1)+1;
    end
    maxFrames = max(framesPerRun);
    nScans = length(framesPerRun);
    nPredictors = size(X,2);
    
    % init 3D X matrix
    oldX = X;
    X = zeros(maxFrames,nPredictors,nScans);
    
    % reshape (allow for different-length runs)
    for run = 1:nScans
        rng = onsets{run}(1):onsets{run}(end);
        X(1:framesPerRun(run),:,run) = oldX(rng,:);
    end
end



return
