function [stat, ces] = glm_contrast(model,active,control,varargin);
% [stat, ces] = glm_contrast(model,active,control,[options]):
%
% For GLM toolbox, perform a statistical test (default t-test)
% between two sets of conditions, and return a set of significance
% statistics as well as contrast effect sizes.
%
% model is a struct produced by the glm m-file. 
% active is a vector specifying positive conditions for the contrast.
% control is a vector specifying negative conditions for the contrast.
% So, if active were [1 3] and control were [2 5], the contrast
% would be [1 3] > [2 5], ignoring condition 4 as well as any other
% conditions.
%
% stat is a matrix of statistical values [default significance level,
% or -log10(p) of t test], one for each voxel the model was applied
% to. ces is the same size as sig, but specifies an estimate of the
% contrast effect size for each voxel.
%
% Options include:
%   'size',[val]:   specify a size to reshape the significance
%                   and ces slices (e.g., for an inplane slice).
%   'test',[char]:  specify the test type to use. Default is 'tm',
%                   for a t test.
%   
%   'weights',[val]: specify points in the time course of a deconvolved
%                    time series to use for the contrast.
%
%   'p':            return stat in units of p, rather than -log10(p).
%   'ln':           return stat in units of -ln(p), rather than -log10(p).
%
%
%
% 02/05 ras.

% params/defaults
test = 'tm';
units = 'log10p';
resz = [];        % optional size to reshape stat for each condition
dof = model.dof;  % degrees of freedom
voxVar = model.voxIndepNoise;      % noise covariance matrix
nh = model.nh;
nPredictors = size(model.betas,2);
nVoxels = size(model.betas,3);
tcWeights = 1:nh; % for deconvolved tcs only

% parse the options
for i = 1:length(varargin)
    if ischar(varargin{i})
        switch lower(varargin{i})
            case {'size','resize','resz','sz'},
                resz = varargin{i+1};
            case {'test','testtype'},
                test = varargin{i+1};
            case {'tcweights','weights'},
                tcWeights = varargin{i+1};
            case {'units'},
                units = varargin{i+1};
            case {'log','log10','log10p'},
                units = 'log10p';
            case {'ln','lnp','naturallog'},
                units = 'lnp';
            case {'p','pval'},
                units = 'pval';

        end
    end
end

betas = model.betas;

% the betas variable needs to be of size
% nVoxelRows x nVoxelCols x nNonNullConditions
% for going into stxgrinder: the first
% two dimensions can be any format (but the product
% of their size must equal the # of voxels). We'll
% use 1D vectors for each cond, 1 x nVoxels,
% then reshape the results later if preferred:
betas = permute(betas, [3 1 2]);
betas = reshape(betas, [1 nVoxels nh*nPredictors]);

% estimated residual variance:
resVar = sum(model.residual.^2) / model.dof; 

% ras 05/09/05:
% remove null condition
if ismember(active, 0),    active = setdiff(active, 0);   end
if ismember(control, 0),   control = setdiff(control, 0); end

% now get restriction matrix:
if isequal(model.type, 'selective averaging')
    RM = glm_restrictionMatrix(test, nh, nPredictors, active, control, ...
                               tcWeights);    
else
    RM = glm_restrictionMatrix(test, nh, nPredictors, active, control);
end


% apply the contrast:
% use the selected test to look up a 
[vSig pSig ces] = er_stxgrinder(test, betas, resVar, voxVar, dof, RM);

if ~isempty(resz) & prod(resz)==prod(size(vSig))
    vSig = reshape(vSig,resz);
    pSig = reshape(pSig,resz);
    ces = reshape(ces,resz);
end

% correct for sign of effect for t tests
if isequal(test,'t') | isequal(test,'tm')
    signMask = ((vSig>=0) - (vSig<0));
    pSig = pSig .* signMask;
end

% Dont let pSig = zero (messes up log10)
iz = find(abs(pSig) < 10^-50);
pSig(iz) = sign(pSig(iz)) * 10^-50;
iz = find(pSig == 0);
pSig(iz) = 10^-50; % Have to do this because sign(0) = 0


% convert to appropriate units
if ~isempty(units)
    switch units
        case 'log10p',
            stat = -log10(abs(pSig)) .*sign(pSig);;
        case 'lnp',
            stat = -log(abs(pSig))  .*sign(pSig);
        case 'pval',
            stat = pSig;
    end
end


return
