function view = computeContrastMap2(view,active,control,contrastName,varargin);
%
%  view = computeContrastMap2(view,active,control,contrastName,[options]);
%
% Compute a statistical contrast map on data for a view, applying
% a GLM if necessary. Updated version, using the rewritten GLM code.
%
% The map is computed off data from the scan group assigned to
% the view's current scan. (You can specify a different scan
% group as an optional argument). The map is added to the 
% view's map field, and saved in the view's data dir under
% the specified contrast name. 
%
% By default, the map represents the -log10(p) value of a statistical
% T test between the active and control conditions. So, a value of
% 2 indicates that the active conditions are greater than the
% control with a significance of 10^-2, a value of 3 is 10^-3, and
% a value of -2 indicates that control > active with a likelihood of 
% 10^-2. Other tests (F test rather than T test) and units are possible
% using the options specified below.
%
%
%
%
% Options include: 
%
%   [add options for varying slice, statistical test, units]
%
%
% ras 06/06
if nargin<1, help(mfilename); return; end

% allow for a user input if the active and control
% conditions aren't specified
if ~exist('active','var') | isempty(active) & ...
	~exist('control','var') | isempty(control)
   prompt={'Enter Active Conditions:','Enter Control Conditions:'};
   def={'1','0'};
   dlgTitle='Compute Contrast Map 2';
   answer=inputdlg(prompt,dlgTitle,1,def);
   active = str2num(answer{1});
   control = str2num(answer{2});
end
scan = viewGet(view,'curScan');

%%%%% parse the options

% default contrast name
if ~exist('contrastName','var') | isempty(contrastName)
    trials = er_concatParfiles(view);
    activeNames = ''; controlNames = '';
    for i = 1:length(active)
        j = find(trials.condNums==active(i));
        activeNames = [activeNames trials.condNames{j}];
    end
    for i = 1:length(control)
        j = find(trials.condNums==control(i));
        controlNames = [controlNames trials.condNames{j}];
    end
    contrastName = sprintf('%sV%s',activeNames,controlNames)
end

% check if a GLM has been run; if not, run one:
checkPath = fullfile(dataDir(view),sprintf('Scan%i',scan),'glmSlice1.mat');
if ~exist(checkPath,'file')
    % no GLM file found -- ask user if they want to run one
    mrGlobals;
    [scans dt] = er_getScanGroup(view,scan);
    names = {dataTYPES.name};
    q = 'A GLM wasn''t found for this slice and scan. ';
    q = [q 'Do you want to run one now? '];
    q = [q sprintf('%s scans %s',names{dt},num2str(scans))];
    resp = questdlg(q,'Compute Contrast Map');
    if isequal(resp,'Yes')
        % compute one
        view = selectDataType(view,dt);
        view = setCurScan(view,scans(1));
        [newDt newScan] = applyGlm(view, scans);
        view  = selectDataType(view, newDt);
        view = setCurScan(view, newScan);
    else
        disp('Aborting computeContrastMap2')
        return
    end
end

% initialize map cell (of size nScans): 
% If another map with the same name has already been
% computed, load this: we'll just plug in the data for
% the current scan:
mapPath = fullfile(dataDir(view),[contrastName '.mat']);
if exist(mapPath, 'file')
    load(mapPath, 'map', 'mapName');
else
    mapName = contrastName;
    map = cell(1, numScans(view));
end

% initalize volume for the map
mapVol = NaN*ones(dataSize(view, scan));

% main loop: load data from each slice and compute map
hwait = waitbar(0,'Computing Contrast Map');
for slice = 1:numSlices(view)
    [model params] = loadGlmSlice(view, slice, scan);
    waitbar((2*(slice-1)+1)/(2*numSlices(view)),hwait);
    mapVol(:, :, slice) = glm_contrast(model, active, control, ...
                                       'size', sliceDims(view));
    waitbar(slice/numSlices(view), hwait);
end
close(hwait);
   
% set new map in view
map{scan} = mapVol;
view = setParameterMap(view, map, mapName);

% save results
if exist(mapPath,'file')
    save(mapPath, 'map', 'mapName', '-append');
else
    save(mapPath, 'map', 'mapName');
end
fprintf('Saved Contrast Map in %s.\n', mapPath);

refreshScreen(view);


return
