function setColorBar(view, hideState, cmapRange)
%
% setColorBar(view, hideState, [cmapRange])
%
% Redraws the horizontal colorbar at the top of the view.
%
% Inputs:
%   cmapRange: range of colormap entries that you want
%            included in the colorbar
%
% djh, 1/98
% Modified from mrColorBar, written by gmb 9/96.
% ras, 09/04 -- Labels what map is shown on the cbar.
%               This looks nicer when the cbar is moved away
%               from the annotation.

if ~exist('cmapRange','var')
   cmapRange=[1 256];
end

cbar = view.ui.colorbarHandle;

% ras 01/06:
% I added some new code in which visual field mapping parameters
% are set, in a new folder mrLoadRet-3.0/Analysis/VisualField.
% If these parameters are set, the colorbar will be set to a circular
% legend image.
% In case you updated cvs without adding the new directory
% (use the "-d" flag), I made this not crash if you can't find the 
% accessor functions.
try 
    params = retinoGetParams(view);
catch
    params = [];
end
  
if strcmp(hideState,'on')
%     if ~isempty(params) & isequal(view.ui.displayMode, 'ph')
%         % Visual-field map: set special legend depending on type
%         if isequal(params.type, 'polar_angle')
%             setColorBarPolarAngle(view, params, cbar);
%         else
%             setColorBarEccentricity(view, params, cbar);
%         end
%     else
        % undo any changes to the cbar position made by the above plots
        pos=get(cbar,'Position'); pos(3:4)=[.6 .03]; set(cbar,'Position',pos);  
        
        % Re-draw the colorbar and set its axes (if xTicks is valid)
        if max(cmapRange) <= 256 
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            % Older, faster way                    %
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            cbarRange = view.ui.cbarRange;
            if (length(cbarRange)>1)
                set(gcf, 'CurrentAxes', cbar);
                image([cbarRange(1) cbarRange(2)], [], [cmapRange(1):cmapRange(2)]);
                set(gca, 'YTick', []);
                set(gca, 'FontSize', 10);
            end
        else
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            % Newer, slower way                    %
            % (But truecolor & 256 colors/grays)   %
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
            height = 10; % pix dimensions of bar
            width = 180; 
            cbarRange = view.ui.cbarRange;
            dispMode = sprintf('%sMode',viewGet(view,'displayMode'));
            cmap = view.ui.(dispMode).cmap;
            cimg = cmapRange(1):cmapRange(2);
            cimg = upSampleRep(cimg,[height width]);
            cimg = ind2rgb(cimg,cmap);
            set(gcf,'CurrentAxes',cbar);
            imshow(cimg);
            ticks = linspace(1,size(cimg,2),4);
            vals = normalize(ticks,cbarRange(1),cbarRange(2));
            for i = 1:length(ticks)
                labels{i} = sprintf('%2.1f',vals(i));
            end
            set(gca,'XTick',ticks,...
                    'XTickLabel',labels,...
                    'YTick',[],...
                    'FontSize',10);
        end
        
        % label what the color bar denotes
        dispMode = viewGet(view,'displayMode');
        switch dispMode
            case 'amp',
                label = 'Fitted Sinusoid Amplitude';
            case 'co',
                label = 'Fitted Sinusoid Coherence';
            case 'ph',
                label = 'Fitted Sinusoid Phase';
            case 'map',
                label = view.mapName;
            otherwise,
                label = '';
        end
        subplot(cbar);
        title(label, 'FontSize',12, 'FontName','Helvetica');

%     end     % if visual field map cbar / regular cbar
    
else
    % remove the title
    set(gcf, 'CurrentAxes', cbar);
    title('');
    
    % Hide or show the color bar
    set(cbar, 'Visible', hideState);
    temp = get(cbar,'Children');
    set(temp, 'Visible', hideState);
end         % if hideState is on

% Return the current axes to the main image
set(gcf,'CurrentAxes',view.ui.mainAxisHandle);

return
% /--------------------------------------------------------------------/ %




% /--------------------------------------------------------------------/ %
function setColorBarPolarAngle(view, params, cbar);
% set the color bar to illustrate the polar angle traversed in this scan
dispMode = sprintf('%sMode', view.ui.displayMode);
nG = view.ui.(dispMode).numGrays;
cmap = view.ui.(dispMode).cmap(nG+1:end,:);
bgColor = get(view.ui.windowHandle, 'Color');

wedge = polarAngleColorBar(params, cmap, 256, bgColor);

% display the image: we'll need to make the color bar axes a bit larger
axes(cbar);
pos=get(cbar,'Position'); pos(3:4)=[.12 .12]; set(cbar,'Position',pos);  
set(cbar, 'Position', pos);
image(wedge);
axis image; axis off;
text(128, 270, 'Polar Angle', 'FontSize', 10, 'FontName', 'Helvetica', ...
    'FontWeight', 'bold', 'HorizontalAlignment', 'center');
return
% /--------------------------------------------------------------------/ %




% /--------------------------------------------------------------------/ %
function setColorBarEccentricity(view, params, cbar);
% set the color bar to illustrate the eccentricity traversed in this scan
dispMode = sprintf('%sMode', view.ui.displayMode);
nG = view.ui.(dispMode).numGrays;
nC = view.ui.(dispMode).numColors;
cmap = view.ui.(dispMode).cmap(nG+1:end,:);

% undo any changes to the cbar position made by the above plots
pos=get(cbar,'Position'); pos(3:4)=[.6 .03]; set(cbar,'Position',pos);  

set(gcf, 'CurrentAxes', cbar);

% rather than show the rings, we'll explicitly show the mapping from
% phase(color) to eccentricity. The eccentricity will be represented 
% by the range of x-values on which the color range is plotted.
% If there's no blank period, this is easy; if there is, it's a 
% little complicated:
if ismember(lower(params.blankPeriod), {'none' 'frequency tagged'})
    xrng = eccentricity([0 2*pi], params);
    image(xrng, [], [nG+1:nG+nC]);
else
    phi = linspace(0, 2*pi, nC);
    vals = eccentricity(phi, params);
    ok = find(vals ~= -1); % -1 phases map to blank period
    yrng = phi([ok(1) ok(end)]);
    xrng = eccentricity(yrng, params);
    
    % add a stretch that shows the range of colors which map to
    % -1 -- e.g., it seems like they started rising during the blank
    % period. This is tricky.
    if isequal(lower(params.blankPeriod), 'start of cycle')
        xrng = [xrng(1)-1 xrng];
    else
        xrng = [xrng xrng(end)+1]; 
    end
    
    image(xrng, [], nG + [1:nC]);
    label = get(gca, 'XTickLabel');
    for i = 1:size(label, 1), tmp{i} = label(i,:); end
    label = tmp; % these last two lines convert from char matrix -> cell
    if isequal(lower(params.blankPeriod), 'start of cycle')
        label{1} = 'Unmapped';
    else
        label{end} = 'Unmapped'; 
    end    
    set(gca, 'XTickLabel', label);
end

% put up the image:
set(gca, 'YTick', [], 'FontSize', 10);
title('Eccentricity, degrees', 'FontSize', 12)
return
% /--------------------------------------------------------------------/ %




% /--------------------------------------------------------------------/ %
function setColorBarEccentricityOld(view, params, cbar);
% set the color bar to illustrate the eccentricity traversed in this scan
dispMode = sprintf('%sMode', view.ui.displayMode);
nG = view.ui.(dispMode).numGrays;
cmap = view.ui.(dispMode).cmap(nG+1:end,:);
% START GUM (need to test the assumptions here are correct, if so
% will modify cmapRing to use retino params)
if (params.startAngle > params.endAngle)
    % inward motion
    fovealPhase = 2*pi * (11/12);
else
    % outward motion
    fovealPhase = 0;
end 
params.doPlot = 0; 
params.background = get(view.ui.windowHandle, 'Color');
wedge = cmapRing(view, fovealPhase, 'b', 256, 0, [.9 .9 .9]) + nG;
wedge(wedge==max(wedge(:))) = nG;
% END GUM

% display the image: we'll need to make the color bar axes a bit larger
axes(cbar);
pos=get(cbar,'Position'); pos(3:4)=[.12 .12]; set(cbar,'Position',pos);  
image(wedge);
axis image; axis off;
text(128, 270, 'Eccentricity', 'FontSize', 10, 'FontName', 'Helvetica', ...
    'FontWeight', 'bold', 'HorizontalAlignment', 'center');
return