function model = rmGridFit(view,params,doPlot);
% rmGridFit - fit retinotopic model
%
% model=rmGridFit(view,params[,doPlot]);
%
% Brute force fitting of predictions based upon 
%   (a) on/off stimulus sequence (if exists)
%   (b) premade receptive fields (rmDefineParams, x, y, sigma)
%   (c) both a and b.
% Negative fits are discarded for b.
% 
% Output is saved in structure model, which should be accessed
% through rmSet and rmGet.
%
%
% 2005/12 SOD: wrote it.


if ieNotDefined('view'),   error('Need view struct'); end;
if ieNotDefined('params'), error('Need params'); end;
if ieNotDefined('doPlot'), doPlot = []; end;

%-----------------------------------
%--- make trends to fit with the model (dc, linear and sinewaves)
%-----------------------------------
% not sure we want to do that here.
[trends, ntrends]  = rmMakeTrends(params);

%-----------------------------------
%--- now loop over slices
%--- but initiate stuff first
%-----------------------------------
switch lower(params.wData(1:3)),
 case {'fig','roi'},
  loopSlices = 1;
 otherwise,
  loopSlices = 1:params.stim(1).nSlices;
end;
nSlices = length(loopSlices);
imsize  = size(params.stim(1).images);

%-----------------------------------
%--- make all predictions first
%-----------------------------------
prediction = rfMakePrediction(params,0);


% go loop over slices
for slice=loopSlices,
  %-----------------------------------
  % Place datasets behind each other. This is a rather crude way of
  % stimultaneously fitting both. Due to this we cannot
  % prewhiten (we could zeropadd/let the trends deal with this/not care). 
  %-----------------------------------
  [data, params] = rmLoadData(view,params,slice);
  rssdata        = sum(data.^2);

  %-----------------------------------
  % initiate stuff on first loop
  %-----------------------------------
  if slice == 1,
    model = initiateModel(params,nSlices,size(data,2),ntrends);
    if isfield(params.stim(1),'stimOnOffSeq'),
      % make stim on/off sequence
      stim = [];
      for ii = 1:length(params.stim),
        stim = [stim; params.stim(ii).stimOnOffSeq];
      end;
    end;
    if strcmp(params.wData,'roi');
      for mm = 1:length(model),
        model{mm} = rmSet(model{mm},'coords',params.coords);
      end;
    end;
    % put in number of data points. Right now this is the same as
    % size(data,1), but this will not be the case if we zero-padd!
    for mm = 1:length(model),
      model{mm} = rmSet(model{mm},'npoints',size(data,1));
    end;
  end;
  
  %-----------------------------------
  % now we extract only the data from that slice and put it in a
  % temporary structure that will be modified throughout.
  %-----------------------------------
  s = extractSlice(model,slice);
  
  % initiateModel fills the rss-field with Infs. We reset them here
  % to a more data-driven maximum value of sum(data.^2)
  % also make all contrasts here
  C = cell(5,1);
  for n=1:length(s),
    findinf           = find(s{n}.rss==Inf);
    s{n}.rss(findinf) = rssdata(findinf);
    % make contrasts
    if n==1 || n==2 || n==4,
        tmpC    = zeros(1,ntrends+1);
        tmpC(1) = 1;
        C{n}    = tmpC;
    else,
        tmpC          = zeros(4,ntrends+2);
        tmpC(:,[1 2]) = [1 1;1 0;0 1;1 -1];
        C{n}          = tmpC;
    end;        
    clear tmpC;
  end;
  
  %-----------------------------------
  %--- fit on/off model:
  %-----------------------------------
  if isfield(params.stim(1),'stimOnOffSeq'),
    %disp(sprintf('[%s]: on/off model fit.',mfilename));
    
    % fit
    [s{4}.t, s{4}.df, ...
     s{4}.rss,s{4}.b] = rmGLM(data,[stim trends],C{4});
  end; 

  
  %-----------------------------------
  %--- fit different receptive fields profiles
  %--- another loop --- and a slow one too
  %-----------------------------------
  tic; progress = 0;
  for n=1:length(params.analysis.x0),
    %-----------------------------------
    % progress monitor (10 dots) and time indicator
    %-----------------------------------
    if round(n/length(params.analysis.x0)*10)>progress,
      if progress==0,
        % print out estimated time left
        if slice==1,
          esttime = toc*10*nSlices;
          if floor(esttime/3600)>0,
            fprintf(1,'[%s]: estimated processing time: %d hours.\n',...
                    mfilename,ceil(esttime/3600));
          else,
            fprintf(1,'[%s]: estimated processing time: %d minutes.\n',...
                    mfilename,ceil(esttime/60));
          end;
        end;
        fprintf(1,'[%s]: grid (x,y,sigma) fit:',mfilename);
      end;
      % progress monitor
      fprintf(1,'.');drawnow;
      progress = progress + 1;
    end;
    
    %-----------------------------------
    %--- now apply glm to fit RF
    %-----------------------------------
    [m.t,m.df,m.rss,m.b]= rmGLM(data,[prediction(:,n) trends],C{2});
    % unconstrained fit
    s{2}                = updateModel(m,s{2},params,n);
    % contrained fit (positive only)
    m.rss               = removeNegatives(m.b,m.rss,1);
    s{1}                = updateModel(m,s{1},params,n);  

    % to visualize search space
    %  if n==1,
    %    s.model2.allrss       = ones(1,size(data,2)).*model.rss;
    %  end;
    %  s.model2.allrss(n)    = model.rss;
  
    %-----------------------------------
    %--- optional plots - probably not functional any more
    %-----------------------------------
    %if ~isempty(doPlot),
    %  figure(5);clf;imagesc(RF(:,:,1));axis image off;colormap gray;
    %  figure(6);clf;
    %  plot(data,'o');hold on;
    %  plot(components*m.b,'r');
      % title(sprintf('rss: %.1f (min %.1f)',model.rss,s.model2.rss));
    %  drawnow;
    %end;
  
    
    %-----------------------------------
    %--- now apply glm to fit RF and on off model
    %-----------------------------------
    if isfield(params.stim(1),'stimOnOffSeq'),
      [m.t,m.df,m.rss,m.b]= rmGLM(data,[prediction(:,n) stim trends],C{6});
      % unconstrained fit
      s{6}                = updateModel(m,s{6},params,n);  
      % contrained fit (positive only)
      m.rss               = removeNegatives(m.b,m.rss,1);
      s{5}                = updateModel(m,s{5},params,n);  
    end;
    
    %-----------------------------------
    %--- try two rf profiles, yet another loop
    %-----------------------------------
    if ~isempty(params.analysis.sigmaRatio),
      %-----------------------------------
      %--- first make all second rf profiles
      %-----------------------------------
      sigmaNew = params.analysis.sigmaRatio.*params.analysis.sigmaMajor(n);
      sigmaNew = sigmaNew(sigmaNew<params.analysis.sigmaRatioMaxVal);
      lSN      = length(sigmaNew);
      if lSN,
        sigmaNew(lSN+1,1) = params.analysis.sigmaRatioMaxVal;
        lSN               = lSN +1;
      else,
        sigmaNew          = params.analysis.sigmaRatioMaxVal;
        lSN               = 1;
      end;
      tmpmatrix           = zeros(lSN,5);
      tmpmatrix(:,[1 2])  = [sigmaNew sigmaNew];
      tmpmatrix(:,[4 5])  = ones(lSN,1)*...
          [params.analysis.x0(n) params.analysis.y0(n)];
      prediction2 = rfMakePrediction(params,tmpmatrix);
      
      for sr = 1:length(sigmaNew),
        %-----------------------------------
        %--- now apply glm to fit
        %-----------------------------------
        [m.t,m.df,m.rss,m.b]= rmGLM(data,...
                                    [prediction(:,n) prediction2(:,sr) trends],...
                                    C{3});
        % unconstrained fit
        s{3}                = updateModel(m,s{3},params,n,sigmaNew(sr));
      end;
    end;
  end;
  
  
  %-----------------------------------
  % now we put back the temporary data from that slice, but first
  %-----------------------------------
  model = putSlice(model,s,slice);
  
  et  = toc;
  fprintf(1,'Done[%.1fmin].\n',et/60);drawnow;
end;
  
% that's it 
return;
%-----------------------------------

%-----------------------------------
function e=removeNegatives(b,e,id);
% loop over beta ids
% we do not allow negative beta values in the fits
% This is important to discuss and comment later.
% The basic problem is that if you have two complementary locations, you
% could fit with a postive beta on the one that drives the signal or a
% negative beta on the portion of the visual field that never sees the
% stimulus. This would produce the same prediction. We don't like that.
if id==0,return;end;
for n=id,
  e(find(b(n,:)<0)) = 1*Inf;
end;
return;
%-----------------------------------
       

%-----------------------------------
function [model]=updateModel(newmodel,model,params,n,s2);
% find rss values smaller than existing ones, ie update
minRssIndex  = find(newmodel.rss < model.rss);
% now update
model.x(minRssIndex)        = params.analysis.x0(n);
model.y(minRssIndex)        = params.analysis.y0(n);
model.s(minRssIndex)        = params.analysis.sigmaMajor(n);
model.rss(minRssIndex)      = newmodel.rss(:,minRssIndex);
model.t(:,minRssIndex)      = newmodel.t(:,minRssIndex);
model.b(:,minRssIndex)      = newmodel.b(:,minRssIndex);
model.df                    = newmodel.df;
if nargin == 5,
  model.s2(minRssIndex)     = s2;
end;
return;
%-----------------------------------

%-----------------------------------
function tmp = extractSlice(model,slice);
% some code to extract slice from model struct and place it
% into temporary struct
f = {'x','y','s','rss','t','b'};

% loop over models
for n=1:length(model),
  
  if n==1 | n==2 | n==4, % retinotopy and full field
    f = {'x','y','s','rss','t'};
  else,
    if n==3,
      f = {'x','y','s','s2','rss'};
    else,
      f = {'x','y','s','rss'};
    end;
    % put all t values in one matrix
    ts         = {'tall','trm','tf','trmf'};
    tmp{n}.t   = zeros(length(ts),size(val,2));
    for fn = 1:length(ts),
      val            = rmGet(model{n},ts{fn});
      tmp{n}.t(fn,:) = val(slice,:);    
    end;
  end;
  
  % for all models
  tmp{n}.desc = rmGet(model{n},'desc');
  tmp{n}.df   = rmGet(model{n},'dfglm');
  for fn = 1:length(f),
    val    = rmGet(model{n},f{fn});
    if ~isempty(val),
      tmp{n}.(f{fn}) = val(slice,:);
    end;
  end;
  
  % put all beta values in one matrix
  val      = rmGet(model{n},'b');
  tmp{n}.b = zeros(size(val,3),size(val,2));
  for fn = 1:size(val,3),
    tmp{n}.b(fn,:) = val(slice,:,fn);
  end;
end;

return;
%-----------------------------------

%-----------------------------------
function model = putSlice(model,tmp,slice);
% some code to put slice info into model struct

% loop over models
for n=1:length(model),
  if n<=2, % retinotopy only
    ftmp = {'x','y','s','rss','t'};
    fput = {'x','y','s','rss','trm'};
    model{n}       = rmSet(model{n},'dfcorr',tmp{n}.df-3); % for x,y,s
  elseif n==4, % full field only
    ftmp = {'x','y','s','rss','t'};
    fput = {'x','y','s','rss','tf'};
    model{n}       = rmSet(model{n},'dfcorr',tmp{n}.df);
  else, % both
    if n==3,      
      ftmp = {'x','y','s','s2','rss'};
      model{n}       = rmSet(model{n},'dfcorr',tmp{n}.df-4); % for x,y,s,s2
    else,         
      ftmp = {'x','y','s','rss'};          
      model{n}       = rmSet(model{n},'dfcorr',tmp{n}.df-3); % for x,y,s
    end;
    fput = ftmp;
    
    % distribute t values
    ts  = {'tall','trm','tf','trmf'};
    for fn = 1:4,
      val           = rmGet(model{n},ts{fn});
      val(slice,:)  = tmp{n}.t(fn,:);
      model{n}      = rmSet(model{n},ts{fn},val);
    end;
  end;
  
  % now get values from model and put in new slice values
  for fn = 1:length(ftmp),
    val             = rmGet(model{n},fput{fn});
    if ~isempty(val),
      val(slice,:)  = tmp{n}.(ftmp{fn});
      model{n}      = rmSet(model{n},fput{fn},val);
    end;
  end;
  model{n}       = rmSet(model{n},'dfglm',tmp{n}.df);

  % distribute beta values
  val      = rmGet(model{n},'b');
  for fn = 1:size(val,3),
    val(slice,:,fn) = tmp{n}.b(fn,:);
  end;
  model{n} = rmSet(model{n},'b',val);
   
end;
return;
%-----------------------------------

%-----------------------------------
function model = initiateModel(params,d1,d2,nt);
% make the model struct with rmSet
fillwithzeros       = zeros(d1,d2);
fillwithzerostrends = zeros(d1,d2,nt);
fillwithinfs        = ones(d1,d2).*Inf;


% retinotopy 
for n=1:3,  
  model{n} = rmSet;
  model{n} = rmSet(model{n},'x'   ,fillwithzeros);
  model{n} = rmSet(model{n},'y'   ,fillwithzeros);
  model{n} = rmSet(model{n},'s'   ,fillwithzeros);
  model{n} = rmSet(model{n},'rss' ,fillwithinfs);
  model{n} = rmSet(model{n},'df'  ,0);
  model{n} = rmSet(model{n},'ntrends',nt);
  if n<3,
    model{n} = rmSet(model{n},'b'   ,zeros(d1,d2,nt+1));
    model{n} = rmSet(model{n},'trm' ,fillwithzeros);
  else,
    model{n} = rmSet(model{n},'s2'  ,fillwithzeros);
    model{n} = rmSet(model{n},'b'   ,zeros(d1,d2,nt+2));
    model{n} = rmSet(model{n},'tf'  ,fillwithzeros);
    model{n} = rmSet(model{n},'trm' ,fillwithzeros);
    model{n} = rmSet(model{n},'tall',fillwithzeros);
    model{n} = rmSet(model{n},'trmf',fillwithzeros);
  end;
end;
model{1} = rmSet(model{1},'desc','2D RF (x,y,sigma) fit (positive only)');
model{2} = rmSet(model{2},'desc','2D RF (x,y,sigma) fit (no constraint)');
model{3} = rmSet(model{3},'desc','Double 2D RF (x,y,sigma) fit (no constraint)');

if isfield(params.stim(1),'stimOnOffSeq'),
  % on off alone
  model{4} = rmSet;
  model{4} = rmSet(model{4},'desc','on/off only fit');
  model{4} = rmSet(model{4},'b'   ,zeros(d1,d2,nt+1));
  model{4} = rmSet(model{4},'rss' ,fillwithinfs);
  model{4} = rmSet(model{4},'df'  ,0);
  model{4} = rmSet(model{4},'tf'  ,fillwithzeros);
  model{4} = rmSet(model{4},'ntrends',nt);

  % both
  for n=5:6,
    model{n} = rmSet;
    model{n} = rmSet(model{n},'x'   ,fillwithzeros);
    model{n} = rmSet(model{n},'y'   ,fillwithzeros);
    model{n} = rmSet(model{n},'s'   ,fillwithzeros);
    model{n} = rmSet(model{n},'b'   ,zeros(d1,d2,nt+2));
    model{n} = rmSet(model{n},'rss' ,fillwithinfs);
    model{n} = rmSet(model{n},'df'  ,0);
    model{n} = rmSet(model{n},'tf'  ,fillwithzeros);
    model{n} = rmSet(model{n},'trm' ,fillwithzeros);
    model{n} = rmSet(model{n},'tall',fillwithzeros);
    model{n} = rmSet(model{n},'trmf',fillwithzeros);
    model{n} = rmSet(model{n},'ntrends',nt);
  end;
  model{5} = rmSet(model{5},'desc',...
                   '2D RF (x,y,sigma) with on/off fit (positive only)');
  model{6} = rmSet(model{6},'desc',...
                   '2D RF (x,y,sigma) with on/off fit (no constraint)');
end;
return;
%-----------------------------------
