function val = rmGet(model,param);
% rmGet - Retrieve data from various retinotopy models
%
% val = rmGet(model,param);
%
% Read the parameters of a retinotopy model. Access to these
% structures should go through this routine and rmSet.
%
% rmGet will return empty ([]) if the parameter is known but not
% defined. In other words, rmGet will only create an error if you 
% ask for a parameter that is not known.
% 
% 2006/01 SOD: wrote it.

if ieNotDefined('model'), error('No model defined.');    end;
if ieNotDefined('param'), error('No parameter defined'); end;

% default output
val = [];
% each of these 'switch case' calls is inbedded in a 'try' loop
% so if that when the requested parameter is not defined, rmGet will return 
% empty and exit gracefully.
switch lower(param),
  % model
 case {'desc','description'}
  try, val = model.description; end;
 case {'x0','x'}
  try, val = model.x0; end;
 case {'y0','y'}
  try, val = model.y0; end;
  
 case {'s','sigma'}
  try, val = (model.sigma.major + model.sigma.minor)./2; end;
 case {'sigmamajor','sigma major'}
  try, val = model.sigma.major; end;
 case {'sigmaminor','sigma minor'}
  try, val = model.sigma.minor; end;
 case {'sigmatheta','sigma theta'}
  try, val = model.sigma.theta; end;
  
 case {'s2','sigma2'}
  try, val = (model.sigma2.major + model.sigma2.minor)./2; end;
 case {'sigma2major','sigma2 major'}
  try, val = model.sigma2.major; end;
 case {'sigma2minor','sigma2 minor'}
  try, val = model.sigma2.minor; end;
 case {'sigma2theta','sigma2 theta'}
  try, val = model.sigma2.theta; end;
 
 case {'r','sr','sigmaratio','sigma ratio'}
  try,  
    val1 = (model.sigma.major + model.sigma.minor)./2;
    val2 = (model.sigma2.major + model.sigma2.minor)./2; 
    val  = val2./val1;
  end;
  
  
  % fit
 case {'b','beta','amp','amplitude'}
  try, val = model.beta; end;
 case {'bcomp1','bcomp2','bcomp3','bcomp4',...
       'amplitude comp1','amplitude comp2',...
       'amplitude comp3','amplitude comp4',}
  compnumber = str2num(lower(param(end)));
  try,
    if     length(size(model.beta)) == 3
      val = model.beta(:,:,compnumber); 
    elseif length(size(model.beta)) == 4
      val = model.beta(:,:,:,compnumber);
    end;
  end;
 
 case {'nt','ntrends','number of trends'},
  try, val = model.beta.ntrends; end;
  
  % t-statistical values
 case {'tf','t fullfield','t-statistic fullfield'}
  try, val = model.t.fullfield; end;
 case {'trm','t retinotopy','t-statistic retinotopy'}
  try, val = model.t.retinotopy; end;
 case {'tall','t all','t-statistic all'}
  try, val = model.t.all; end;
 case {'trmf','t rmf','t-statistic retinotopy vs fullfield',...
       't-statistic retinotopy minus fullfield'}
  try, val = model.t.ret_vs_full; end;
 case {'t-statistic fullfield vs retinotopy',...
       't-statistic fullfield minus retinotopy'}
  try, val = model.t.ret_vs_full.*-1; end;
 case {'t'}
  if isfield(model.t,'all'),
    try, val = model.t.all; end;
  elseif isfield(model.t,'retinotopy'),
    try, val = model.t.retinotopy; end;
  elseif isfield(model.t,'fullfield'),
    try, val = model.t.fullfield; end;
  end;
  
  % p-value
 case {'pf','p fullfield','p-statistic fullfield'}
  try, val = rmT2P(model.t.fullfield,model.df.glm,'p'); end;
 case {'prm','p retinotopy','p-statistic retinotopy'}
  try, val = rmT2P(model.t.retinotopy,model.df.glm,'p'); end;
 case {'pall','p all','p-statistic all'}
  try, val = rmT2P(model.t.all,model.df.glm,'p'); end;
 case {'prmf','p rmf','p-statistic retinotopy vs fullfield',...
       'p-statistic retinotopy minus fullfield'}
  try, val = rmT2P(model.t.ret_vs_full,model.df.glm,'p'); end;
 case {'p-statistic fullfield vs retinotopy',...
       'p-statistic fullfield minus retinotopy'}
  try, val = rmT2P(model.t.ret_vs_full,model.df.glm,'p').*-1; end;
 case {'p'}
  if isfield(model.t,'all'),
    try, val = rmT2P(model.t.all,model.df.glm,'p'); end;
  elseif isfield(model.t,'retinotopy'),
    try, val = rmT2P(model.t.retinotopy,model.df.glm,'p'); end;
  elseif isfield(model.t,'fullfield'),
    try, val = rmT2P(model.t.fullfield,model.df.glm,'p'); end;
  end;
  
  % log10p-value
 case {'log10pf','log10p fullfield','log10p-statistic fullfield'}
  try, val = rmT2P(model.t.fullfield,model.df.glm,'log10p'); end;
 case {'log10prm','log10p retinotolog10py','log10p-statistic retinotopy'}
  try, val = rmT2P(model.t.retinotopy,model.df.glm,'log10p'); end;
 case {'log10pall','log10p all','log10p-statistic all'}
  try, val = rmT2P(model.t.all,model.df.glm,'log10p'); end;
 case {'log10prmf','log10p rmf','log10p-statistic retinotopy vs fullfield',...
      'log10p-statistic retinotopy minus fullfield'}
  try, val = rmT2P(model.t.ret_vs_full,model.df.glm,'log10p'); end;
 case {'log10p-statistic fullfield vs retinotopy',...
       'log10p-statistic fullfield minus retinotopy'}
  try, val = rmT2P(model.t.ret_vs_full,model.df.glm,'log10p').*-1; end;
 case {'log10p'}
  if isfield(model.t,'all'),
    try, val = rmT2P(model.t.all,model.df.glm,'log10p'); end;
  elseif isfield(model.t,'retinotopy'),
    try, val = rmT2P(model.t.retinotopy,model.df.glm,'log10p'); end;
  elseif isfield(model.t,'fullfield'),
    try, val = rmT2P(model.t.fullfield,model.df.glm,'log10p'); end;
  end;
  
  % other model fits
 case {'rss','residual sum of squares'}
  try, val = model.rss; end;
 
 case {'df','degrees of freedom'}
  if isfield(model.df,'glm'),
    try, val = model.df.glm; end;
  elseif isfield(model.df,'glm_corrected'),
    try, val = model.df.glm_corrected; end;
  end;
 case {'dfglm','degrees of freedom glm'}
  try, val = model.df.glm; end;
 case {'dfcorr','degrees of freedom model corrected'}
  try, val = model.df.glm_corrected; end;
   
  % derived spatial values
 case {'ecc','eccentricity'}
  try, [tmp val] = cart2pol(model.x0,model.y0); end;
 case {'pol','polar-angle'}
  try, val       = cart2pol(model.x0,model.y0) + pi; end;
 
  % others
 case {'coords','roicoords'}
  try, val       = model.roi_coordinates; end;
 case {'n','npoints','number of data points'}
  try, val = model.npoints; end;
  
 otherwise,
  error(sprintf('[%s]:Unknown parameter: %s.',mfilename,param));
end;

return;
