function view = averageTSeries(view, scanList, typeName, annotation)
%
%  view = averageTSeries(view, [scanList], [typeName], [annotation])
%
% Form an average of the time series scans specified by the
% scanList input parameter. Saves results in the 'Averages' data type,
% unless an alternate data type is specifed by the 'typeName' argument.
%
% The optional 'annotation' argument specifies the annotation (the text
% that pops up for the scan in mrVista) for the scan. Default is to use 
% the text 'Average of [data type] [scans] scans'.
%
% Uses the current dataType of the view to determine which tSeries
% to average.
%
% dbr, 3/99
% djh, 2/28/2001, Reimplemented for mrLoadRet-3.0
% arw, 08/28/02, Added option to set new data type name.
% ras, 12/17/05, Added optional new annotation.
mrGlobals

% default input args
if ieNotDefined('scanList'),   scanList = er_selectScans(view); end
if ieNotDefined('typeName'),   typeName='Averages';             end
if ieNotDefined('annotation'),   
    annotation = sprintf('Average of %s scans: %s', ...
                          getDataTypeName(view), ...
                          num2str(scanList));
end

% create the new data type if it doesn't already exist
if ~existDataType(typeName),     addDataType(typeName);         end

checkScans(view,scanList);

% Open a hidden view and set its dataType to 'Averages'
switch view.viewType
    case 'Inplane'
        hiddenView = initHiddenInplane;
    case 'Volume'
        hiddenView = initHiddenVolume;
    case 'Gray'
        hiddenView = initHiddenGray;
    case 'Flat'
        hiddenView = initHiddenFlat(viewDir(view));
end
hiddenView = selectDataType(hiddenView,typeName);

% Set dataTYPES.scanParams so that new average scan has the same params as
% the 1st scan on scanList.
newScanNum = numScans(hiddenView)+1;
ndataType = hiddenView.curDataType;
dataTYPES(ndataType).scanParams(newScanNum) = ...
    dataTYPES(view.curDataType).scanParams(scanList(1));
dataTYPES(ndataType).blockedAnalysisParams(newScanNum) = ...
    dataTYPES(view.curDataType).blockedAnalysisParams(scanList(1));
dataTYPES(ndataType).eventAnalysisParams(newScanNum) = ...
    dataTYPES(view.curDataType).eventAnalysisParams(scanList(1));
dataTYPES(ndataType).scanParams(newScanNum).annotation = annotation;

saveSession

% Get the tSeries directory for this dataType 
% (make the directory if it doesn't already exist).
tseriesdir = tSeriesDir(hiddenView);

% Make the Scan subdirectory for the new tSeries (if it doesn't exist)
scandir = fullfile(tseriesdir,['Scan',num2str(newScanNum)]);
if ~exist(scandir,'dir')
    mkdir(tseriesdir,['Scan',num2str(newScanNum)]);
end

% Double loop through slices and scans in scanList
nAvg = length(scanList);
% *** check that all scans have the same slices
waitHandle = waitbar(0,'Averaging tSeries.  Please wait...');
nSlices = length(sliceList(view,scanList(1)));
for iSlice = sliceList(view,scanList(1));
    % For each slice...
    % disp(['Averaging scans for slice ', int2str(iSlice)])
    for iAvg=1:nAvg
        iScan = scanList(iAvg);
        tSeries = loadtSeries(view, iScan, iSlice);
        bad = isnan(tSeries);
        tSeries(bad) = 0;
        if iAvg > 1;
            tSeriesAvg = tSeriesAvg + tSeries;
            nValid = nValid + ~bad;
        else
            tSeriesAvg = tSeries;
            nValid = ~bad;
        end
    end
    tSeriesAvg = tSeriesAvg ./ nValid;
    tSeriesAvg(nValid == 0) = NaN;
    savetSeries(tSeriesAvg,hiddenView,newScanNum,iSlice);
    waitbar(iSlice/nSlices);
end
close(waitHandle);

% Loop through the open views, switch their curDataType appropriately, 
% and update the dataType popups
INPLANE = resetDataTypes(INPLANE,ndataType);
VOLUME  = resetDataTypes(VOLUME,ndataType);
FLAT    = resetDataTypes(FLAT,ndataType);

disp('Done Averaging tSeries.')

return;
% /-----------------------------------------------------------/ %




% /-----------------------------------------------------------/ %
function checkScans(view,scanList)
%
% Check that all scans in scanList have the same slices, numFrames, cropSizes
for iscan = 2:length(scanList)
    if find(sliceList(view,scanList(1)) ~= sliceList(view,scanList(iscan)))
        mrErrorDlg('Can not average these scans; they have different slices.');
    end
    if (numFrames(view,scanList(1)) ~= numFrames(view,scanList(iscan)))
        mrErrorDlg('Can not average these scans; they have different numFrames.');
    end
    if find(sliceDims(view,scanList(1)) ~= sliceDims(view,scanList(iscan)))
        mrErrorDlg('Can not average these scans; they have different cropSizes.');
    end
end
return;
