function motion = motionComp(view, mcView, scan, nSmooth, baseFrame)
%
% motion = motionComp(view,scan[, nSmooth, baseFrame])
%
% Robust 3D rigid body motion compensation
%
% If you change this function make parallel changes in:
%   betweenScanMotComp, inplaneMotionComp
%
% djh & on, 2000
% Ress, 3/05: Added temporal smoothing option to deal with low SNR
% functionals and artifacts. Moved default base frame to center of time
% series.

if ~exist('scan','var')
    scan = getCurScan(view);
end

if exist('nSmooth', 'var')
    nSmooth = 2*fix(nSmooth/2) + 1;
else
    nSmooth = 1;
end

slices = sliceList(view,scan);
nSlices = length(slices);
nFrames = numFrames(view,scan);
dims = sliceDims(view,scan);
motion = zeros(2, nFrames);

if ~exist('baseFrame','var') | isempty(baseFrame)
    baseFrame = round(nFrames/2);
end

% Load tSeries from all slices into one big array
volSeries = zeros([dims(1) dims(2) nSlices nFrames]);
midX = [dims/2 nSlices/2]';
waitHandle = waitbar(0,'Loading tSeries from all slices. Please wait...');
for slice=slices
    waitbar(slice/nSlices);
    ts = loadtSeries(view,scan,slice);
    for frame=1:nFrames
        volSeries(:,:,slice,frame) = reshape(ts(frame,:),dims);
    end
end
close(waitHandle)

% Get base volume.  Other frames will be motion compensated to this one.
baseMin = baseFrame - fix(nSmooth/2);
if baseMin < 1, baseMin = 1; end
baseMax = baseFrame + fix(nSmooth/2);
if baseMax > nFrames, baseMax = nFrames; end
baseVol = mean(volSeries(:,:,:,baseMin:baseMax), 4);

% Do motion estimation/compensation for each frame.
warpedVolSeries = zeros(size(volSeries));
% filling the base frame
warpedVolSeries(:,:,:,baseFrame) = baseVol;
% if the number of slices is too small, repeat the first and last slice
% to avoid running out of data (the derivative computation discards the
% borders in z, typically 2 slices at the begining and 2 more at the end)
if size(baseVol,3)<=8
    baseVol = cat(3,baseVol(:,:,1),baseVol(:,:,1),baseVol,...
        baseVol(:,:,end),baseVol(:,:,end));
end

waitHandle = waitbar(0,'Computing motion estimates. Please wait...');
for frame = 1:nFrames
    waitbar(frame/nFrames)
    if (frame~=baseFrame)
        frameMin = frame - fix(nSmooth/2);
        if frameMin < 1, frameMin = 1; end
        frameMax = frame + fix(nSmooth/2);
        if frameMax > nFrames, frameMax = nFrames; end
        vol = mean(volSeries(:,:,:,frameMin:frameMax), 4);
        % if the number of slices is too small, repeat the first and last slice
        % to avoid running out of data (the derivative computation discards the
        % borders in z, typically 2 slices at the begining and 2 more at the end)
        if size(vol,3)<=8
            vol = cat(3, vol(:,:,1),vol(:,:,1),vol,...
                vol(:,:,end),vol(:,:,end));
        end
        
        M = estMotionIter3(baseVol,vol,2,eye(4),1,1); % rigid body, ROBUST
        % warp the volume putting an edge of 1 voxel around to avoid lost data
        warpedVolSeries(:,:,:,frame) = warpAffine3(volSeries(:,:,:,frame),M,NaN,1);
        midXp = M(1:3, 1:3) * midX; % Rotational motion
        motion(1, frame) = sqrt(sum((midXp - midX).^2));
        motion(2, frame) = sqrt(sum(M(1:3, 4).^2)); % Translational motion
    end
end
close(waitHandle)

% Save warped tSeries to tSeries.dat
tSeries = zeros(size(ts));
numPixels = size(tSeries,2);
waitHandle = waitbar(0,'Saving tSeries. Please wait...');
for slice=slices
  waitbar(slice/nSlices);
  for frame=1:nFrames
    tSeries(frame,:) = reshape(warpedVolSeries(:,:,slice,frame),[1 numPixels]);
  end
  savetSeries(tSeries, mcView, scan, slice);
end
close(waitHandle)

return
