function AdjustSliceTiming(scans, typeName)
% AdjustSliceTiming([scans, typeName])
%
% Uses cubic-spline interpolation to offset the time series based on their
% slice position and number-of-interleaves. Specify which scans to adjust
% in the currently selected INPLANE view, or use the input scans parameter.
% Creates a new datatype named "Timed" by default or use the typeName input
% to specify.
%
% Ress, 02/05
% Rory, 01/06: adjusted so that the input and output scan numbers don't
% have to match; will append new time-corrected scans to an existing data
% type.
mrGlobals;

% get the selected view (need to have an open inplane for this)
selectedINPLANE = viewSelected('inplane');
view = INPLANE{selectedINPLANE};
if ~exist('scans', 'var'), scans = selectScans(view); end
srcDt = viewGet(view, 'curDataType');

% Set up new datatype for timing correction:
hiddenView = initHiddenInplane;
if (~exist('typeName', 'var') | isempty(typeName)), typeName = 'Timed'; end
if ~existDataType(typeName), addDataType(typeName); end
hiddenView = selectDataType(hiddenView, existDataType(typeName));
ndataType = hiddenView.curDataType;

% Get the tSeries directory for this dataType
% (make the directory if it doesn't already exist).
tsDir = tSeriesDir(hiddenView);

for i = 1:length(scans)
    scan = scans(i);
    
    % initialize a slot for the new scan
    hiddenView = initScan(hiddenView, typeName, [], {srcDt scan});
    outScan = numScans(hiddenView); % # of scan in the new data type
    
    % Make the Scan subdirectory for the new tSeries (if it doesn't exist)
    scanDir = fullfile(tsDir, ['Scan',int2str(outScan)]);
    if ~exist(scanDir, 'dir')
        mkdir(tsDir, ['Scan' int2str(outScan)]);
    end
    
    % get params for slice-timing correction
    scanParams = mrSESSION.functionals(scan);
    slices = scanParams.slices;
    refSlice = round(mean(slices));
    nSlices = scanParams.reconParams.slquant;
    nShots = scanParams.reconParams.nshots;
    deltaFrame = 1 / nSlices / nShots;
    
    % main loop: loop across slices, doing spline interpolation
    wH = waitbar(0, ['Adjusting scan ' int2str(scan)]);
    iS = 0;
    for slice=slices
        iS = iS + 1;
        ts = loadTSeries(view, scan, slice);
        if slice ~= refSlice
            nFrames = size(ts, 1);
            % Replicate first and last frames to deal with extrapolation:
            ts = [ts(1, :); ts; ts(nFrames, :)];
            tBaseRef = (0:nFrames+1);
            tBase = (1:nFrames) + deltaFrame * (refSlice - slice);
            ts = spline(tBaseRef, ts', tBase)';
        end
        savetSeries(ts, hiddenView, outScan, slice);
        waitbar(iS/length(slices), wH);
    end
    close(wH);
end

saveSession;

return
