function subdata = getCurDataROI(view, fieldname, scan, roi)
%
% subdata = getCurDataROI(view, <fieldname>, <scan>, <roi>)
%
% Pulls out co, amp, ph, or map for given scan and ROI.  Returns
% a vector that corresponds to roi.coords.  
%
% scan: integer
% fieldname: 'co', 'amp', or 'ph'
% roi: can be an ROI struct, name of ROI, index into view's
%      ROIs, or 3xN array of (y,x,z) coordinates
%
% Treats INPLANE and VOLUME/FLAT differently.
%
% For INPLANE, gets the indices of the roi.coords.  Returns vector
% of subdata for those indices.  
%
% For VOLUME, data is stored as nVoxels x nScans.  Intersect
% the roi.coords with view.coords, and pull out the corresponding
% indices of the data vector, then construct the subdata vector
% with NaNs where there's no data.
%
% djh, 7/98
%
% djh, 2/19/2001
% - tSeries and corAnal no longer interpolated to inplane size
%
% ras, 01/19/2004
% - fixed a bug with the upSampleFactor in the case where the upSample factor
%   is different in different directions.
%
% ras, 1/16/2006
% - made roi input be flexible as to how it's specified.
if notDefined('view'), view = getSelectedInplane; end
if notDefined('fieldname'), fieldName = view.ui.displayMode; end
if notDefined('scan'), scan = viewGet(view, 'curScan'); end
if notDefined('roi'), roi = viewGet(view, 'curRoi'); end

roi = tc_roiStruct(view, roi);

switch view.viewType
    
case 'Inplane'
    % Pull out data for this scan
    data = getCurData(view,fieldname,scan);
    % Construct subdata for voxels in ROI
    % Need to divide the roi.coords by the upSample factor because the
    % data are no longer interpolated to the inplane size.
    rsFactor = upSampleFactor(view,scan);
    if length(rsFactor)==1
        roi.coords(1:2,:)=round(roi.coords(1:2,:)/rsFactor(1));
    else
        roi.coords(1,:)=round(roi.coords(1,:)/rsFactor(1));
        roi.coords(2,:)=round(roi.coords(2,:)/rsFactor(2));
    end
    indices = coords2Indices(roi.coords,dataSize(view,scan));
    if ~isempty(data)
        subdata = data(indices);    
    else
        subdata = NaN*ones(size(indices));
    end
    
case {'Volume','Gray'}
    % Pull out data for this scan
    tmp = getfield(view,fieldname);
    data = tmp{scan};
    
    % Intersect the two sets of indices, filling in data for the
    % intersection, NaNs for ROI voxels where there's no data.
    [inter,ROIIndices,viewIndices] = intersectCols(roi.coords,view.coords);
    subdata = NaN*ones([1,size(roi.coords,2)]);
    if ~isempty(data)
        subdata(ROIIndices) = data(viewIndices);
    end
    
case 'Flat'
    % Pull out data for this scan
    data = getCurData(view,fieldname,scan);
    % Construct subdata for voxels in ROI
    indices = coords2Indices(roi.coords,viewSize(view));
    if ~isempty(data)
        subdata = data(indices);   
    else
        subdata = NaN*ones(size(indices));
    end
end
    
return;
    
    