function v2gMap = mrmMapVerticesToGray(vertexCoords, grayNodes, mmPerVox, grayEdges)
%
% v2gMap = mrmMapVerticesToGray(vertices, grayNodes, mmPerVox, [grayEdges]);
%
% Finds a map between the mesh vertices and layer 1 gray nodes.
% To see the  coordinate of a gray matter node nearest to a vertexCoord, we
% can use
%    
%      grayNodes(1:3,v2gMap(idx))
%
% If the grayEdges are also passed in, then we'll compute the mapping for
% all gray layers. Layer 1 mapping will be in the first column, layer 2 in
% the second, etc. As always, vertices with no mapping will be set to 0.
%
% HISTORY:
%  2003.09.16 RFD (bob@white.stanford.edu)
%  2004.05.21 RFD: now uses Dan Merget's nearpoints to find a proper
%  mapping. No more hacks! Aside from being a bit faster, the result should
%  be substantially more accurate.
%  2005.07.26 RFD: We now return a mapping for all gray layers, if
%  grayEdges is passed in.
%  2005.08.05 RFD: all gray layer mapping is turned off by default, until
%  we figure out how to make it go faster.
%  2005.10.26 GB: Temptative of speed improvement in the main loop

if ieNotDefined('mmPerVox'), error('Voxel size (mm per vox) is required.'); end;

vertexCoords = double(vertexCoords);
grayNodes = double(grayNodes);
grayEdges = double(grayEdges);

prefs = mrmPreferences;
if(strcmp(prefs.layerMapMode,'all'))
    mapToAllLayers = true;
else
    mapToAllLayers = false;
end

% This is now a real distance threshold, in mm. For each mesh vertex, the
% algorithm will find the nearest gray coord. If the nearest coord is >
% distThresh from the vertex, then that vertex gets no mapping ('0'). 
distThresh = 4;

% The gray coordinates are in voxels in the vAnatomy file.  This scales
% them into real physical (mm) coordinates.  And transposes them.
grayCoords = grayNodes([1,2,3], :);
grayCoords = [grayCoords(1,:).*mmPerVox(1); ...
        grayCoords(2,:).*mmPerVox(2); ...
        grayCoords(3,:).*mmPerVox(3) ]';

% Transposes these mesh coordinates, which were already built in real
% physical coordinates.  
% Major comments needed here.
vertexCoords = vertexCoords' + 1;

% Mask out non-layer 1 nodes so that they are not found
grayCoords(find(grayNodes(6,:)~=1),:) = -9999;

% Some day, this will be better.  Even sooner, the logic will be commented.
%v2gMap = zeros(size(vertexCoords(1,:)));
%v2gMap = zeros(1,length(vertexCoords));

[v2gMap, sqDist] = nearpoints(vertexCoords', double(grayCoords'));
v2gMap(sqDist > distThresh^2) = 0;
v2gMap = int32(v2gMap);

if(mapToAllLayers && exist('grayEdges','var') && ~isempty(grayEdges))
    % h = waitbar(0, 'Mapping vertices to all gray layers...');
    % The following (from mrFlatMesh) might be promising...
    %conMat = makeGrayConMat(grayNodes, grayEdges, 0);
    %L1 = find(grayNodes(6,:)==1);
    %L2 = find(grayNodes(6,:)==2);
    %mappedIndices = findNearestConnected(grayNodes', L2, L1, conMat);

%     % do the other layers
%     L1 = find(grayNodes(6,:)==1);
%     L2 = find(grayNodes(6,:)==2);
%     L2toL1 = zeros(size(L2));
%     curLayerNum = 2;
%     for(ii=1:length(L2))
%         offset = grayNodes(5,L2(ii));
%         numConnected = grayNodes(4,L2(ii));
%         neighbors = grayEdges(offset:offset+numConnected);
%         lowerNeighbors = neighbors(grayNodes(6,neighbors)==curLayerNum-1);
%         if(~isempty(lowerNeighbors))
%             ind = nearpoints(grayNodes(1:3,L2(ii)), grayNodes(1:3,lowerNeighbors));
%             L2toL1(ii) = lowerNeighbors(ind);
%         end
%     end

    layer1Nodes = v2gMap;
    curValid = double(layer1Nodes) > 0;
    numLayers = max(grayNodes(6,:));
    n = length(curValid);
    curLayerNum = 2;

    % GB 2005.10.26
    % We map from higher layers to lower layers,
    % selecting the closest connected node in the lower layer. This way we
    % will avoid multiply-connected 

    noNeighborsDown = 0;
    noNeighbor = 0;
    noNeighbor2 = 0;
    
    v2gMapTemp = zeros(numLayers,size(grayNodes,2));
    
    h = waitbar(0, 'Computing between layers connections...');
    progress = 0;
    for iLayer = numLayers:-1:curLayerNum
        midprogress = 0;
        
        curNodes = find(grayNodes(6,:) == iLayer);
        for iNode = curNodes
            offset = grayNodes(5,iNode);
            numConnected = grayNodes(4,iNode);
            if numConnected == 0
                noNeighbor = noNeighbor + 1;
                continue
            end
            neighbors = grayEdges(offset:offset + numConnected - 1);
            neighbors(grayNodes(6,neighbors) ~= (iLayer - 1)) = [];
            distance = sum(grayNodes(1:3,neighbors).^2,1);
            [value,nearest] = min(distance);
            if ~isempty(nearest)
                v2gMapTemp(iLayer,iNode) = neighbors(nearest);
            elseif length(nearest) > 1
                neighbors = neighbors(nearest);
                distanceIndex = abs(neighbors - iNode);
                [value,nearest] = min(distanceIndex);
                v2gMapTemp(iLayer,iNode) = neighbors(nearest);
            else
                neighbors = grayEdges(offset:offset + numConnected - 1);
                neighbors(grayNodes(6,neighbors) < iLayer - 1) = [];
                nextNeighbors = [];
                neighborsDown = [];
                
                while isempty(neighborsDown) && ~isequal(neighbors,nextNeighbors)
                    neighbors = union(nextNeighbors,neighbors);
                    nextNeighbors = [];
                    for iNeighbor = 1:length(neighbors)
                        offset = grayNodes(5,neighbors(iNeighbor));
                        numConnected = grayNodes(4,neighbors(iNeighbor));
                        nextNeighbors = [nextNeighbors grayEdges(offset:offset+numConnected-1)];
                    end
                    nextNeighbors = unique(nextNeighbors);
                    neighborsDown = nextNeighbors(grayNodes(6,nextNeighbors) == (iLayer - 1));
                end
                
                if isempty(neighborsDown)
                    noNeighbor2 = noNeighbor2 + 1;
                else
                    distance = sum(grayNodes(1:3,neighborsDown).^2,1);
                    [value,nearest] = min(distance);
                    if length(nearest) == 1
                        v2gMapTemp(iLayer,iNode) = neighborsDown(nearest(1));
                    else
                        neighborsDown = neighborsDown(nearest);
                        distanceIndex = abs(neighborsDown - iNode);
                        [value,nearest] = min(distanceIndex);
                        v2gMapTemp(iLayer,iNode) = neighborsDown(nearest);
                    end
                end
                noNeighborsDown = noNeighborsDown + 1;
            end
        end
        progress = progress + 1/(numLayers - curLayerNum + 1);
        waitbar(progress/2,h);
    end
    
    indices = 2*ones(1,size(v2gMap,2));
    indexNodes = cell(1,size(grayNodes,2));
    for curNode = find(curValid)
        indexNodes{v2gMap(1,curNode)} = [indexNodes{v2gMap(1,curNode)} curNode];
    end
    
    h = waitbar(1/2, h, 'Creating connections table...');
    progress = 0;
    for iNode = 1:size(grayNodes,2)
        if iNode > (progress + 1/10)*size(grayNodes,2)
            progress = progress + 1/10;
            waitbar(0.5 + progress/2);
        end
        
        curLayer = grayNodes(6,iNode);
        curNode = iNode;
        for iLayer = curLayer:-1:2
            if curNode == 0
                break
            end
            curNode = v2gMapTemp(iLayer,curNode);
        end
        if (curNode == 0) || (curNode == iNode)
            continue
        end
        %v2gMapCell{curNode} = [v2gMapCell{curNode};iNode];
        indexNode = indexNodes{curNode};
        if isempty(indexNode)
            continue
        end
        
        v2gMap(indices(indexNode),indexNode) = iNode;
        indices(indexNode) = indices(indexNode) + 1;
        
    end
    close(h);
 
end

% [goodLocInd, loc] = ismember(round(vertexCoords/4), round(grayCoords/4), 'rows');
% v2gMap(goodLocInd) = loc(goodLocInd);
% [goodLocInd, loc] = ismember(round(vertexCoords/2), round(grayCoords/2), 'rows');
% v2gMap(goodLocInd) = loc(goodLocInd);
% [goodLocInd, loc] = ismember(vertexCoords, grayCoords, 'rows');
% v2gMap(goodLocInd) = loc(goodLocInd);

% We spent a couple of days and concluded that the delaunay calls here
% sometimes return gray points that are very far away from the vertex
% points. We don't think it should, and we will try to get this code
% working.  In the mean time, we reverted to Bob's original algorithm/hack.
% 
% Map each vertex to a gray node.
% v2gMap = zeros(size(vertexCoords(1,:)));
% grayTesselation = delaunay3(grayCoords(:,1),grayCoords(:,2),grayCoords(:,3));
% OUTVAL = Inf; 
% [v2gMap,distance] = dsearchn(grayCoords,grayTesselation,vertexCoords);

% grayTesselation = delaunay3(grayCoords(:,1),grayCoords(:,2),grayCoords(:,3));
% OUTVAL = Inf; [v2gMap,distance] = dsearchn(grayCoords,grayTesselation,vertexCoords,OUTVAL);
% sum(v2gMap == OUTVAL)

% If a vertex is more than 2 voxels from data, set it to 0
% l = (distance > 3); 
% v2gMap(l) = 0;

return;

% %
% f = grayNodes(6,:) == 1;
% figure; plot3(vertexCoords(:,1)+1,vertexCoords(:,2)+1,vertexCoords(:,3)+1,'.')
% hold on; plot3(grayCoords(f,1),grayCoords(f,2),grayCoords(f,3),'r.')
% 
% figure; hist(distance(:),100)
% 
% % To see the nearest grayCoord to a vertexCoord, we have the vCoord index
% % and we look at grayCoord(v2gMap(idx))