function makeAnalyzeFromIfiles(ifileDir, outFileName)
% makeAnalyzeFromIfiles(ifileDir, [outFileName])
%
% ifileDir must include the filename prefix- eg. 'spgr1/I' for I-files
% that look like 'I.001', I.002, ...'.
%
% outFileName should not include '.hdr' or '.img'- just the base name.
% It defaults to a reasonable file name in the current working directory.
%
% The resulting Analyze file will be oriented according to 
% the default Analyze convention- 'transverse unflipped' (orient code '0' in 
% the .hdr file). That is, the image will be reoriented so that left-right is 
% along the x-axis, anterior-posterior is along the y-axis, superior-inferior 
% is along the z-axis, and the leftmost, anterior-most, superior-most point is 
% at 0,0,0 (which, for Analyze, is the lower right-hand corner of the last slice). 
%
% To help you find the 0,0,0 point, a 4-pixel rectange is drawn there with a pixel
% value equal to the maximum image intensity.
%
% The reorientation involves only cannonical rotations and slice flips- no
% interpolation is performed. Also, note that the reorientation depends on info
% in the GE I-file header.
%
% REQUIRES:
%   * SPM99 utilities
%   * Stanford Anatomy and filter functions
%
% SEE ALSO: loadAnalyze, saveAnalyze, analyze2mrGray, mrGray2Analyze
%
% HISTORY:
%   2002.05.31 RFD (bob@white.stanford.edu) wrote it.

if(~exist('ifileDir','var') | isempty(ifileDir))
    help(mfilename);
    return;
end
if(~exist('outFileName','var') | isempty(outFileName))
    [p,f] = fileparts(ifileDir);
    outFileName = fullfile(pwd, f);
end

disp(['Converting ifiles in ',ifileDir,' to Analyze format...']);
[su_hdr,ex_hdr,se_hdr,im_hdr] = GE_readHeader([ifileDir,'.001']);
mmPerVox = [im_hdr.pixsize_X, im_hdr.pixsize_Y, im_hdr.slthick];
% The coordinates actually specify the location of the *center* of the slice.
% Thus, the nSlices dimension is actually im_hdr.slthick thicker than it seems to 
% be if you just do the simple, obvious thing (as we do below). The proper fix is to
% add im_hdr.slthick/2 to each side of the volume. The quick fix is to make nSlices
% nSlice-1.
%nSlices = se_hdr.se_numimages;
d = dir([ifileDir,'*']);
% for DTI data, we can't rely on the header to tell us the number of slices
% so we just count the image files and grab them all.
nSlices = length(d);
imDim = [im_hdr.dim_X, im_hdr.dim_Y, nSlices-1];
% to locate the volume in physical space, we need GE ccords from another slice
[su_hdr,ex_hdr,se_hdr,im_hdr2] = GE_readHeader([ifileDir,sprintf('.%03d',nSlices)]);
% Now we find the Affine transform to make the voxels isotropic and orient the
% volume to the talairach standard (upper left pixel is right, anterior, superior).
trcRas1 = [im_hdr.trhc_R, im_hdr.trhc_A, im_hdr.trhc_S];
brcRas1 = [im_hdr.brhc_R, im_hdr.brhc_A, im_hdr.brhc_S];
tlcRas1 = [im_hdr.tlhc_R, im_hdr.tlhc_A, im_hdr.tlhc_S];
trcRas2 = [im_hdr2.trhc_R, im_hdr2.trhc_A, im_hdr2.trhc_S];
brcRas2 = [im_hdr2.brhc_R, im_hdr2.brhc_A, im_hdr2.brhc_S];
tlcRas2 = [im_hdr2.tlhc_R, im_hdr2.tlhc_A, im_hdr2.tlhc_S];
% To guarantee that we can find any reference point, we also need to specify the 
% bottom left corner (blc), which is simple to compute via vector addition (making
% trc the origin):
blcRas1 = trcRas1 + (tlcRas1-trcRas1) + (brcRas1-trcRas1);
blcRas2 = trcRas2 + (tlcRas2-trcRas2) + (brcRas2-trcRas2);

volRas = [tlcRas1; blcRas1; trcRas1; brcRas1; tlcRas2; blcRas2; trcRas2; brcRas2];
% We need to remember the xyz axis mappings for the GE 'tlc', 'brc', etc. convention
volXyz = [0,0,0; 1,0,0; 0,1,0; 1,1,0; 0,0,1; 1,0,1; 0,1,1; 1,1,1];

% Now we need to find the correct rotation & slice reordering to bring it into our standard space,
% which is axials with tlc, slice 1 as the most left, most anterior, most superior point.
% We do this by finding the most right, most anterior, and most superior point (ras), the
% most left, most anterior, and most superior point (las), etc.
d = sqrt((-1000-volRas(:,1)).^2 + (1000-volRas(:,2)).^2 + (1000-volRas(:,3)).^2);
las = find(min(d)==d); las = las(1);
d = sqrt((1000-volRas(:,1)).^2 + (1000-volRas(:,2)).^2 + (1000-volRas(:,3)).^2);
ras = find(min(d)==d); ras = ras(1);
d = sqrt((-1000-volRas(:,1)).^2 + (-1000-volRas(:,2)).^2 + (1000-volRas(:,3)).^2);
lps = find(min(d)==d); lps = lps(1);
d = sqrt((-1000-volRas(:,1)).^2 + (1000-volRas(:,2)).^2 + (-1000-volRas(:,3)).^2);
lai = find(min(d)==d); lai = lai(1);

volCoords = [volRas(las,:),1; volRas(lps,:),1; volRas(lai,:),1; volRas(ras,:),1;];
% The GE convention is that negative values are left, posterior and inferior.
% However, we want our pixels to go left-right, anterior-posterior, and superior-inferior.
% Thus, we flip the sign of the AP and SI axes: 
volCoords(:,1:3) = volCoords(:,1:3)*-1;

% The following will tell us how to map the GE axes to our standard axes.
ydim = find(abs(volXyz(las,:)-volXyz(lps,:)));
zdim = find(abs(volXyz(las,:)-volXyz(lai,:)));
xdim = find(abs(volXyz(las,:)-volXyz(ras,:)));
imDimScaled = imDim .* mmPerVox;
mmPerVoxNew = [mmPerVox(xdim), mmPerVox(ydim), mmPerVox(zdim)];

% Map GE axes into our xyz axes.
volCoords = [volCoords(:,xdim), volCoords(:,ydim), volCoords(:,zdim), volCoords(:,4)];

% This should map A-P to y axis, L-R to x-axis, and S-I to z axis.
% If you want a diferent cannonical view, you *should* only need to change this line.
stdCoords = [0,0,0,1; 0,imDimScaled(ydim),0,1; 0,0,imDimScaled(zdim),1; imDimScaled(xdim),0,0,1];

% The following will produce an affine transform matrix that tells us how to transform
% GE's space to our standard space (ras at tlc of slice 1).
std2vol = stdCoords \ volCoords;
% Extract rotation (& scale) matrix. We have set things up so that the scales should be 1.
std2volRot = round(std2vol(1:3,1:3));
% to use this xform matrix, do: volCoords = [stdCoords,1]*std2vol
% Note that we have constructed this transform matrix so that it will only involve cannonical
% rotations. We did this by specifying corresponding points from cannonical locations (see
% stdCoords and volCoords).
disp(volCoords);
disp(stdCoords);
disp(std2vol);
disp(std2volRot);

img = makeCubeIfiles(ifileDir, imDim(1:2), [1:nSlices]);

% We use shortcuts to apply the transform. Since all rotations are cannonical, we can achieve
% them efficiently by swapping dimensions with 'permute'.
% The dimension permutation logic:
xdim = find(abs(std2volRot(1,:))==1);
ydim = find(abs(std2volRot(2,:))==1);
zdim = find(abs(std2volRot(3,:))==1);
img = permute(img, [xdim, ydim, zdim]);
% insert a marker at 0,0,0 (should be left, anterior, superior)
img(1,1,1) = max(img(:));
img(2,1,1) = img(1,1,1);
img(1,2,1) = img(1,1,1);
img(2,2,1) = img(1,1,1);

% Now do any necessary mirror flips (indicated by negative rotation matrix values).
% Note- we actually flip if the rotation is *not* negative, since Analyze format 
% seems to want it that way.
if(std2volRot(1,xdim)>0)
    % flip each slice ud (ie. flip along matlab's first dimension, which is our x-axis)
    for(jj=1:size(img,3))
        img(:,:,jj) = flipud(squeeze(img(:,:,jj)));
    end
end
if(std2volRot(2,ydim)>0)
    % flip each slice lr(ie. flip along matlab's second dimension, which is our y-axis)
    for(jj=1:size(img,3))
        img(:,:,jj) = fliplr(squeeze(img(:,:,jj)));
    end
end   
if(std2volRot(3,zdim)>0)
    % reorder slices
    for(jj=1:size(img,1))
        img(jj,:,:) = fliplr(squeeze(img(jj,:,:)));
    end
end  

disp(['Saving ',outFileName,'.hdr and ',outFileName,'.img ...']);
hdr = saveAnalyze(img, outFileName, mmPerVoxNew);

return;
