
import icp
from numpy import *
from math import sin,cos,pi


def strlst2float(d):
    """Utility function to convert a list of strings, each containing the same number of whitespace seperated floats,
into a numpy array."""
    pts = zeros( (len(d), len(d[0].split())), float)
    for i in xrange(len(d)):
        pts[i,:] = array([float(z) for z in d[i].split()])
    return pts

def load_hs_data(filename):
    """Load in the coordinates of the head shape from polhemus data in 4D's hs_file.txt"""
    
    f = open(filename)
    d = f.readlines()
    f.close()
    
    head_coils = strlst2float(d[6:11])
    head_shape = strlst2float(d[13:])
    
    return (head_shape, head_coils)


def load_scalp_data(filename):
    """Load data from an ascii file with three columns."""
    
    f = open(filename)
    
    num_pts = int(f.readline())     # not used here
     
    pts = reshape( fromfile(f, float, sep=' '), (-1, 3))

    f.close()
    
    return pts


def get_rot_mat(a, b, g):
    """Return the rotation matrix for the given angles (degrees)."""
    
    degreesToRadians = pi / 180
    
    alpha = a * degreesToRadians    # Rotation angle around X-axis
    beta = b * degreesToRadians     # Rotation angle around Y-axis
    gamma = g * degreesToRadians    # Rotation angle around Z-axis
    
    R = empty( (3,3), float)
    R[0,0] = cos(beta)*cos(gamma)
    R[0,1] = cos(gamma)*sin(alpha)*sin(beta)-cos(alpha)*sin(gamma)
    R[0,2] = cos(alpha)*cos(gamma)*sin(beta)+sin(alpha)*sin(gamma)
    R[1,0] = cos(beta)*sin(gamma)
    R[1,1] = cos(alpha)*cos(gamma)+sin(alpha)*sin(beta)*sin(gamma)
    R[1,2] = -cos(gamma)*sin(alpha)+cos(alpha)*sin(beta)*sin(gamma)
    R[2,0] = -sin(beta)
    R[2,1] = cos(beta)*sin(alpha)
    R[2,2] = cos(alpha)*cos(beta)

    return R


#fixed_pts = load_scalp_data('/mnt/mridata/R1025/20060327113034_STRUCTURAL/505_FL_E_GW3D_YNICT1A_SCALP.txt')
fixed_pts = load_scalp_data('/mnt/mridata/R1019/20060817133704_STRUCTURAL/505_FL_E_GW3D_YNICT1B_SCALP.txt')

#head_shape, head_coils = load_hs_data('/mnt/megdata/R1025/P1014e/06%11%03@10:39/1/hs_file.txt')
head_shape, head_coils = load_hs_data('/mnt/megdata/R1019/P1001/07%02%08@15:10/1/hs_file.txt')

from ynic.meg.utils import *
from ynic.meg.framework_utils import *
from random import randint

def coreg(fixed_pts, head_shape, head_coils, num_iter=10, verbose=False):
    
    # Initial rough alignment, appropriate for YNiC structural MRI protocol
    R = get_rot_mat(0, 0, 90)
    t = array([88.0, 145.0, 145.0])     # half the field of view of the structural scans
    sc = 1000       # MRI scan uses units of mm, 4D MEG uses m
    
    # NB icp routine modifies moving_pts in place
    moving_pts = head_shape.copy()
    T0, cp, n, err = icp.icp(moving_pts, fixed_pts, R, t, sc, verbose=0)
    
    
    Rj = 10
    tj = 10
    
    for i in range(num_iter):
    
        # add a small jitter to last result
        R = get_rot_mat(randint(-Rj,Rj), randint(-Rj,Rj), randint(-Rj,Rj))
        t = array([randint(-tj,tj),randint(-tj,tj),randint(-tj,tj)])
        sc = 1
    
        T, cp, n, err = icp.icp(moving_pts, fixed_pts, R, t, sc, verbose=0)
    
        T0 = dot(T, T0)     # compose successive transforms
        
        if (i == 0) or (err < min_err):
            T_full = T0.copy()
            min_err = err
    
        if verbose:
            print i,n,err
    
    if verbose:
        print min_err
        print T_full
    
    head_shape_mri = apply_transform(T_full, head_shape)
    head_coils_mri = apply_transform(T_full, head_coils)

    return (T_full, min_err, head_shape_mri, head_coils_mri)



